/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
* FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
* SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
* CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
* CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
****************************************************************************/

#include "WindowSystems/WaylandBaseWindowSystem.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"
#include "Log.h"
#include "Layer.h"
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <linux/fb.h>
#include <fcntl.h>
#include <libudev.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <iomanip>
#include <fstream>
#include <errno.h>
#include <sys/timerfd.h>
#include "WindowSystems/WaylandServerinfoServerProtocol.h"
#include "InputManager.h"
#include "wayland-1.0-event-loop-imports.h"

#ifdef WITH_WAYLAND_SYSTEMD_SYNC
#include <systemd/sd-daemon.h>
#endif

// Define update period for FPS calculations (seconds)
#define FPS_UPDATE_PERIOD      5
#define REPAINT_COUNT_UPDATE_PERIOD      1.0f
#define REPAINT_COUNT_TO_SWITCH      58.0f

static float timediff(struct timeval& t1, struct timeval& t2)
{
    return (float)(t2.tv_sec - t1.tv_sec)
           + 0.000001 * (float)(t2.tv_usec - t1.tv_usec);
}

static inline void
timespecSub(struct timespec *result,
            const struct timespec *time1, const struct timespec *time2)
{
    result->tv_sec = time1->tv_sec - time2->tv_sec;
    result->tv_nsec = time1->tv_nsec - time2->tv_nsec;
    if (result->tv_nsec < 0) {
        result->tv_sec--;
        result->tv_nsec += NANOSEC_PER_SEC;
    }
}

static inline int64_t
timespecToNsec(const struct timespec *time)
{
    return (int64_t)time->tv_sec * NANOSEC_PER_SEC + time->tv_nsec;
}

static inline float
timespecToSec(const struct timespec *time1, const struct timespec *time2)
{
    return (float)(time2->tv_sec - time1->tv_sec)
            + 0.000000001 * (float)(time2->tv_nsec - time1->tv_nsec);
}

extern "C" {
struct serverinfo
{
    struct wl_resource base;
    WaylandBaseWindowSystem* windowSystem;
};

struct serverinfoClient
{
    struct wl_client *client;
    uint connectionId;
    struct wl_list link;
};

struct compositor_region
{
    struct wl_resource *resource;
};

void WaylandBaseWindowSystem::serverinfoIFCreateConnection(struct wl_client *client, struct wl_resource *resource)
{
    struct serverinfo* deliver = (struct serverinfo*)resource->data;

    // creates new connection id and store it in memory
    struct serverinfoClient* clientPair = (struct serverinfoClient*)malloc(sizeof *clientPair);

    clientPair->client = client;
    clientPair->connectionId = deliver->windowSystem->m_manageConnectionId;
    deliver->windowSystem->m_manageConnectionId++;

    wl_list_init(&clientPair->link);
    wl_list_insert(&deliver->windowSystem->m_connectionList, &clientPair->link);

    // send native client handle to this client.
    // by protocol default, this information is not needed.
    wl_resource_post_event(resource, SERVERINFO_CONNECTION_ID, clientPair->connectionId);
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Created connection, "
              "client=" << client << ", "
              "ID=" << clientPair->connectionId << ", "
              "for client=" << clientPair->client << ", "
              "resource=" << resource);
}

struct serverinfo_interface g_serverinfoImplementation = {
    WaylandBaseWindowSystem::serverinfoIFCreateConnection,
};

void WaylandBaseWindowSystem::bindServerinfo(struct wl_client *client, void *data, uint32_t version, uint32_t id)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "client=" << client << ", data=" << data << ", "
              "version=" << version << ", ID=" << id);
    wl_client_add_object(client, &serverinfo_interface, &g_serverinfoImplementation, id, data);
}

void WaylandBaseWindowSystem::createServerinfo(WaylandBaseWindowSystem* windowSystem)
{
    struct serverinfo* serverInfo;

    serverInfo = (struct serverinfo*)malloc(sizeof *serverInfo);
    if (NULL == serverInfo)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Failed to alloc serverinfo, "
                    "window system=" << windowSystem);
        return;
    }

    serverInfo->base.object.interface = &serverinfo_interface;
    serverInfo->base.object.implementation = (void(**)(void)) &g_serverinfoImplementation;
    serverInfo->base.client = NULL;
    serverInfo->base.data = NULL;
    serverInfo->windowSystem = windowSystem;

    m_serverInfoGlobal = wl_display_add_global(windowSystem->m_wlDisplay,
                                                &serverinfo_interface,
                                                serverInfo,
                                                WaylandBaseWindowSystem::bindServerinfo);
    if (NULL == m_serverInfoGlobal)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "failed wl_display_add_global" << ", "
                    "added server info=" << serverInfo << ", "
                    "window system=" << windowSystem);
        free(serverInfo);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "wl_display_add_global, "
                  "added server info=" << serverInfo << ", "
                  "server info global=" << m_serverInfoGlobal << ", "
                  "window system=" << windowSystem);
    }

    m_serverInfo = (void*)serverInfo;
}

/*current implementation of wl_output interface will provide the width and height
 * which were passed as command line parameters(-w,-h) of layer manager*/
void WaylandBaseWindowSystem::bindOutputInfo(wl_client *client, void *data, uint32_t version, uint32_t id)
{
    struct wl_resource *resource;
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "data=" << data << ", "
              "version=" << version << ", "
              "ID=" << id << ", "
              "window system=" << windowSystem);

    if (windowSystem)
    {
        resource = wl_client_add_object(client, &wl_output_interface, NULL, id, data);
        wl_output_send_geometry(resource, 0, 0, 100, 100, WL_OUTPUT_SUBPIXEL_UNKNOWN, "", "", WL_OUTPUT_TRANSFORM_NORMAL);
        wl_output_send_mode(resource,
                WL_OUTPUT_MODE_CURRENT | WL_OUTPUT_MODE_PREFERRED,
                windowSystem->getWindowWidth(),
                windowSystem->getWindowHeight(), 60 * 1000);
    }
    else
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Passed data is NULL, "
                    "client=" << client << ", "
                    "data=" << data << ", "
                    "version=" << version << ", "
                    "ID=" << id);
    }

}

void WaylandBaseWindowSystem::createOutputinfo(WaylandBaseWindowSystem* windowSystem)
{
    wl_display_add_global(windowSystem->m_wlDisplay,
                          &wl_output_interface,
                          windowSystem,
                          WaylandBaseWindowSystem::bindOutputInfo);
}

int WaylandBaseWindowSystem::redrawCheckHandler(void *data)
{
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);
    if (windowSystem)
    {
        for (std::map<uint, struct repaint_data*>::iterator it = windowSystem->m_repaintData.begin();
             it != windowSystem->m_repaintData.end();
             it++)
        {
            struct repaint_data* repaintData;
            repaintData = it->second;
            if (repaintData->bRepaintNeeded) {
                windowSystem->scheduleRepaint(repaintData);
            }
        }
    }
    return 1;
}

int WaylandBaseWindowSystem::finishFrameHandler(void *data)
{
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);
    if (windowSystem)
    {
        for (std::map<uint, struct repaint_data*>::iterator it = windowSystem->m_repaintData.begin();
             it != windowSystem->m_repaintData.end();
             it++)
        {
            struct repaint_data* repaintData;
            repaintData = it->second;
            windowSystem->scheduleRepaint(repaintData);
        }
    }
    return 1;
}

int WaylandBaseWindowSystem::doRepaintTimer(int screenID)
{
    std::map<uint, struct repaint_data*>::iterator match;
    struct repaint_data* repaintData;

    match = m_repaintData.find(screenID);

    if(match == m_repaintData.end()) {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "ScreenId " << screenID <<
                    " is not present in m_repaintData map");
        return 0;
    }
    else {
        repaintData = match->second;
    }

    WaylandBaseWindowSystem::repaintTimerHandler((void*)repaintData);

    return 1;
}

int WaylandBaseWindowSystem::fakeVsyncHandler(void *data)
{
    struct native_frame_callback* cb;
    struct native_frame_callback* cnext;
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);

    /*dispatch frame callback every fake vsync*/
    wl_list_for_each_safe(cb, cnext, &windowSystem->m_listFrameCallback, link)
    {
        wl_callback_send_done(&cb->resource, getTime());
        wl_resource_destroy(&cb->resource);
    }
    return true;
}

int WaylandBaseWindowSystem::screenShotHandler(void *data)
{
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);
    if (windowSystem)
    {
        windowSystem->Screenshot();
    }
    else
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Failed to get WaylandBaseWindowSystem from data=" << data);
    }
    return 1;
}

void WaylandBaseWindowSystem::switchCurrentScreen(void *data, uint32_t screenID)
{
    WaylandBaseWindowSystem* windowSystem =
            static_cast<WaylandBaseWindowSystem*>(data);

    windowSystem->m_currentScreenID = screenID;
}

void WaylandBaseWindowSystem::checkForRenderModelSwitching(struct repaint_data* repaintData)
{
    WaylandBaseWindowSystem* windowSystem =
            static_cast<WaylandBaseWindowSystem*>(repaintData->baseWindowSystem);
    struct timespec currentTime;
    float timeElapsed;

    if (repaintData == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "repaintData is NULL");
        return;
    }

    repaintData->repaintCount++;

    clock_gettime(CLOCK_MONOTONIC, &currentTime);

    if (repaintData->lastTime.tv_sec == 0 && repaintData->lastTime.tv_nsec == 0)
        repaintData->lastTime = currentTime;

    timeElapsed = timespecToSec(&repaintData->lastTime, &currentTime);

    if (timeElapsed >= REPAINT_COUNT_UPDATE_PERIOD)
    {
        float repaintsPerSec = repaintData->repaintCount / timeElapsed;

        repaintData->lastTime = currentTime;

        LOG_INFO("WaylandBaseWindowSystem", "repaint info" <<
                ", screenID=" << repaintData->screenId <<
                ", repaintCount=" << repaintData->repaintCount <<
                ", repaintsPerSec=" << repaintsPerSec <<
                ", timeElapsed=" << timeElapsed <<
                ", repaintCountToSwitch=" << REPAINT_COUNT_TO_SWITCH);

        if (repaintsPerSec < REPAINT_COUNT_TO_SWITCH)
        {
            if (repaintData->bUseRepaintTimer)
                LOG_INFO("WaylandBaseWindowSystem", "Switching to normal model" <<
                        ", screenID=" << repaintData->screenId <<
                        ", repaintsPerSec=" << repaintsPerSec <<
                        ", repaintCountToSwitch=" << REPAINT_COUNT_TO_SWITCH);

            repaintData->bUseRepaintTimer = false;
        }
        else
        {
            if (!repaintData->bUseRepaintTimer)
                LOG_INFO("WaylandBaseWindowSystem", "Switching to timer rendering" <<
                         ", screenID=" << repaintData->screenId <<
                         ", repaintsPerSec=" << repaintsPerSec <<
                         ", repaintCountToSwitch=" << REPAINT_COUNT_TO_SWITCH);

            repaintData->bUseRepaintTimer = true;
        }

        repaintData->repaintCount = 0;
    }
}

int WaylandBaseWindowSystem::repaintTimerHandler(void *data)
{
    struct repaint_data* repaintData = (struct repaint_data*)data;
    WaylandBaseWindowSystem* windowSystem =
            static_cast<WaylandBaseWindowSystem*>(repaintData->baseWindowSystem);
    struct timespec start_time;
    struct timespec gone_time;
    bool bscreenWiseRendering = false;
    int repaintTime;

    LOG_DEBUG("WaylandBaseWindowSystem",
              "data=" << data);

    clock_gettime(CLOCK_MONOTONIC, &start_time);

    repaintData->lastSubmittedSurface = NULL;

    if (!repaintData->bRepaintNeeded) {
        repaintData->bRepaintScheduled = false;
        repaintData->bRepaintTimerTriggered = false;
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "bRepaintNeeded not set for screenID=" << repaintData->screenId);
        return 1;
    }

    if (windowSystem)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "starts redraw for screenID=" << repaintData->screenId);

        windowSystem->m_pScene->lockScene();
        windowSystem->switchCurrentScreen(repaintData->baseWindowSystem, repaintData->screenId);
        repaintData->bRedrawDone = false;
        windowSystem->finishFrame();
        repaintData->bRepaintScheduled = false;
        repaintData->bRepaintTimerTriggered = false;
        windowSystem->m_pScene->unlockScene();

        LOG_DEBUG("WaylandBaseWindowSystem",
                  "finish redraw for screenID=" << repaintData->screenId);
    }

    clock_gettime(CLOCK_MONOTONIC, &gone_time);

    timespecSub(&gone_time, &gone_time, &start_time);

    repaintTime = (timespecToNsec(&gone_time)) / 1000000; /* floor */

    if (repaintData->repaintWindowPeriodMS &&
        (repaintTime >= repaintData->repaintWindowPeriodMS))
    {
        LOG_INFO("WaylandBaseWindowSystem",
                 "repaintTime takes longer than expected, " <<
                 "RepaintTime=" << repaintTime << ", " <<
                 "repaintWindowPeriodMS=" << repaintData->repaintWindowPeriodMS);
    }

    bscreenWiseRendering = windowSystem->isScreenWiseRenderingNeeded();

    if (repaintData->bRedrawDone && bscreenWiseRendering)
        windowSystem->checkForRenderModelSwitching(repaintData);

    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");

    return 1;
}

} // extern "C"

WaylandBaseWindowSystem::WaylandBaseWindowSystem(ICommandExecutor& executor, const char* displayname, int width, int height, Scene* pScene, InputManager* pInputManager)
: BaseWindowSystem(pScene, pInputManager)
, m_wlDisplay(NULL)
, renderThread(0)
, run_lock()
, sem_run()
, graphicSystem(NULL)
, m_wlShm(NULL)
, m_serverInfoGlobal(NULL)
, m_serverInfo(NULL)
, m_wlCompositorGlobal(NULL)
, m_initialized(false)
, m_takeScreenshot(ScreenShotNone)
, m_displayname(displayname)
, m_success(false)
, m_systemState(IDLE_STATE)
, m_manageConnectionId(256)
, m_screenShotFile()
, m_screenShotScreenID(0)
, m_screenShotSurfaceID(0)
, m_screenShotLayerID(0)
, m_debugMode(false)
, m_error(false)
, m_width(width)
, m_height(height)
, m_bUseFrameTimer(true)
, m_finishFrameTimer(NULL)
, m_screenShotTimer(NULL)
, m_redrawCheckTimer(NULL)
, m_vnceventsflushTimer(NULL)
, m_fakeVysncSimulator(NULL)
, m_listFrameCallback()
, m_inputEvent(NULL)
, m_commandExecutor(executor)
, m_connectionList()
, m_currentScreenID(-1)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, Creating WaylandBaseWindowSystem, "
              "width=" << width << ", height=" << height << ", "
              "display name=" << m_displayname);

    LmScreenList screenList = m_pScene->getScreenList();
    for (LmScreenListIterator iter = screenList.begin();
         iter != screenList.end();
         ++iter)
    {
        m_screenData.insert(std::pair<uint, struct screen_data*>((*iter)->getID(),
                                                                 new struct screen_data()));
    }

    // init and take mutex, so windowsystem only does init phase until mutex is released again
    pthread_mutex_init(&run_lock, NULL);
    pthread_mutex_lock(&run_lock);

    sem_init(&sem_run, 0, 0);
}

WaylandBaseWindowSystem::~WaylandBaseWindowSystem()
{
    if (m_inputEvent)
        delete m_inputEvent;

    for (std::map<uint, struct screen_data*>::iterator it = m_screenData.begin();
        it != m_screenData.end();
        it++)
    {
        delete it->second;
    }

    sem_destroy(&sem_run);
}

void WaylandBaseWindowSystem::printDebug()
{
    // print stuff about layerlist
    std::stringstream debugmessage;
    debugmessage << "Layer:  ID |   X  |   Y  |   W  |   H  | Al. \n";

    LayerList list = m_pScene->getCurrentRenderOrder(0);

    // loop the layers
    LayerListConstIterator iter = list.begin();
    LayerListConstIterator iterEnd = list.end();

    for (; iter != iterEnd; ++iter)
    {
        Rectangle dest = (*iter)->getDestinationRegion();
        debugmessage << "            " << std::setw(4) << (*iter)->getID() << " " << std::setw(3) << dest.x << " " << std::setw(3) << dest.y << " " << std::setw(3) << dest.width << " " << std::setw(3) << dest.height << " " << std::setw(3) << (*iter)->opacity << "\n";

        debugmessage << "    Surface:  ID |Al.|  SVP: X |  Y |  W |  H     DVP:  X |  Y |  W |  H \n";

        // loop the surfaces of within each layer
        SurfaceList surfaceList = (*iter)->getAllSurfaces();
        SurfaceListIterator surfaceIter = surfaceList.begin();
        SurfaceListIterator surfaceIterEnd = surfaceList.end();

        for (; surfaceIter != surfaceIterEnd; ++surfaceIter)
        {
            Rectangle src = (*surfaceIter)->getSourceRegion();
            Rectangle dest = (*surfaceIter)->getDestinationRegion();
            debugmessage << "                        " << std::setw(4) << (*surfaceIter)->getID() << " " << std::setprecision(3) << (*surfaceIter)->opacity << " " << std::setw(3) << src.x << " " << std::setw(3) << src.y << " " << std::setw(3) << src.width << " " << std::setw(3) << src.height << " " << std::setw(3) << dest.x << " " << std::setw(3) << dest.y << " " << std::setw(3) << dest.width << " " << std::setw(3) << dest.height << "\n";
        }
    }
    LOG_DEBUG("WaylandBaseWindowSystem", debugmessage.str());
}

Surface* WaylandBaseWindowSystem::getSurfaceFromNativeSurface(struct native_surface* nativeSurface)
{
    // go though all surfaces
    const std::map<unsigned int, Surface*> surfaces = m_pScene->getAllSurfaces();
    for (std::map<unsigned int, Surface*>::const_iterator currentS = surfaces.begin(); currentS != surfaces.end(); ++currentS)
    {
        Surface* currentSurface = (*currentS).second;
        if (!currentSurface)
        {
            continue;
        }
        WaylandPlatformSurface* nativePlatform = (WaylandPlatformSurface*)currentSurface->platform;
        if (!nativePlatform)
        {
            LOG_WARNING("WaylandBaseWindowSystem","nativePlatform is NULL");
            continue;
        }
        if (nativePlatform->connectionId != nativeSurface->connectionId)
        {
            continue;
        }
        if (nativePlatform->surfaceId != nativeSurface->surface.resource.object.id)
        {
            continue;
        }
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Got surface, native surface ID="
                  << nativeSurface->surface.resource.object.id << ", "
                  "native surface=" << nativeSurface << ", "
                  "surface=" << currentSurface);
        return currentSurface;
    }
    LOG_WARNING("WaylandBaseWindowSystem",
                "Could not find surface, native surface ID="
                << nativeSurface->surface.resource.object.id << ", "
                "native surface=" << nativeSurface);
    return NULL;
}

void WaylandBaseWindowSystem::scheduleRepaintForSurface(struct native_surface* nativeSurface)
{
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);
    std::map<uint, struct repaint_data*>::iterator match;
    struct repaint_data* repaintData = NULL;
    bool bscreenWiseRendering = false;
    bool brepaintScheduled = false;

    Surface* surface = windowSystem->getSurfaceFromNativeSurface(nativeSurface);
    bscreenWiseRendering = windowSystem->isScreenWiseRenderingNeeded();

    /*
     * To check the need for rendering one screen at a time.
     * Now the framecallbacks postings are made properly for
     * Gal2DWindowSystem, So timer-fd are created per screen.
     * Repaint will be triggered from these timers for every
     * screen just before the VBlank. In all the other WindowSystems
     * this is not supported, so isScreenWiseRenderingNeeded always
     * returns false and only a single instance of repaintData
     * is maintained.
     *
     * If the native_surface is not attached to any ILM Surface/Layer,
     * Repaint is scheduled with the first available repaintData
     * to ensure the frame callbacks are posted
     */
    if (bscreenWiseRendering && surface
        && (-1 != surface->getContainingLayerId()))
    {
        int layerID = surface->getContainingLayerId();
        LmScreenList screenList = windowSystem->m_pScene->getScreenList();
        std::list<struct repaint_data*> screensToRedraw;

        LOG_INFO("WaylandBaseWindowSystem",
                  "surfaceID=" << surface->getID() << ", " <<
                  "layerID=" << layerID);

        for (LmScreenListIterator iter = screenList.begin();
                iter != screenList.end();
                ++iter)
        {
            if (windowSystem->m_pScene->isLayerPresentInScreen((*iter)->getID(),
                    layerID)) {
                LOG_INFO("WaylandBaseWindowSystem",
                          "screenID=" << (*iter)->getID());

                match = windowSystem->m_repaintData.find((*iter)->getID());
                if (match != windowSystem->m_repaintData.end())
                {
                    struct native_frame_callback* cb;

                    repaintData = match->second;

                    if (!repaintData->bUseRepaintTimer)
                    {
                        LOG_DEBUG("WaylandBaseWindowSystem", "Added to fakeVsync list CB="
                                << &nativeSurface->pending.frame_callback_list);

                        wl_list_insert_list(windowSystem->m_listFrameCallback.prev,
                                &nativeSurface->pending.frame_callback_list);
                    }
                    else
                    {
                        LOG_DEBUG("WaylandBaseWindowSystem", "Added to repaint list CB="
                                << &nativeSurface->pending.frame_callback_list);

                        wl_list_insert_list(repaintData->listFrameCallback.prev,
                                &nativeSurface->pending.frame_callback_list);
                    }

                    wl_list_init(&nativeSurface->pending.frame_callback_list);

                    if (!repaintData->bScreenDamaged) {
                        LayerList layers = m_pScene->getCurrentRenderOrder(repaintData->screenId);
                        repaintData->bScreenDamaged = graphicSystem->needsRedraw(layers);
                    }

                    screensToRedraw.push_back(repaintData);

                    if (repaintData->lastSubmittedSurface == surface) {
                        LOG_INFO("WaylandBaseWindowSystem","Surface=" << surface->getID() <<
                                 " last submitted frame is dropped");
                    }

                    repaintData->lastSubmittedSurface = surface;

                    brepaintScheduled = true;
                }
            }
        }

        for (std::list<struct repaint_data*>::iterator cur = screensToRedraw.begin();
             cur != screensToRedraw.end();
             cur++)
        {
            windowSystem->scheduleRepaint(*cur);
        }
    }

    if (!brepaintScheduled)
    {
        match = windowSystem->m_repaintData.begin();
        repaintData = match->second;

        wl_list_insert_list(windowSystem->m_listFrameCallback.prev,
                &nativeSurface->pending.frame_callback_list);
        wl_list_init(&nativeSurface->pending.frame_callback_list);

        windowSystem->scheduleRepaint(repaintData);
    }
}

struct native_surface*
WaylandBaseWindowSystem::getNativeSurfaceFromSurface(Surface* surface)
{
    struct native_surface *nativeSurface = NULL;

    WaylandPlatformSurface* platformSurface = static_cast<WaylandPlatformSurface*>(surface->platform);
    if (!platformSurface)
    {
        LOG_WARNING("WaylandBaseWindowSystem", 
                    "No platform surface assigned, "
                    "surface ID=" << surface->getID() << ", "
                    "surface=" << surface);
        return NULL;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem", 
                  "Platform surface assigned, "
                  "wayland platform surface=" << platformSurface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "surface=" << surface);
    }

    wl_list_for_each(nativeSurface, &m_nativeSurfaceList, link)
    {
        if ((nativeSurface->connectionId == platformSurface->connectionId) &&
            (nativeSurface->surface.resource.object.id == platformSurface->surfaceId))
        {
            LOG_DEBUG("WaylandBaseWindowSystem", 
                      "Native surface found, "
                      "wayland platform surface=" << platformSurface << ", "
                      "native surface=" << nativeSurface << ", "
                      "native surface ID="
                      << nativeSurface->surface.resource.object.id << ", "
                      "native surface connection ID="
                      << nativeSurface->connectionId << ", "
                      "surface ID=" << surface->getID() << ", "
                      "surface=" << surface);
            return nativeSurface; // FOUND
        }
    }

    return NULL;
}

void WaylandBaseWindowSystem::checkForNewSurfaceNativeContent()
{
    m_pScene->lockScene();
    LayerList layers = m_pScene->getCurrentRenderOrder(0);
    for (LayerListConstIterator current = layers.begin(); current != layers.end(); current++)
    {
        SurfaceList surfaces = (*current)->getAllSurfaces();
        for (SurfaceListConstIterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
        {
            if ((*currentS)->hasNativeContent())
            {
                allocatePlatformSurface(*currentS);
            }
            else // While we are at it, also cleanup any stale native content
            {
                deallocatePlatformSurface(*currentS);
            }
        }
    }
    m_pScene->unlockScene();
}

struct timeval tv0;
struct timeval tv0_forRender;


void WaylandBaseWindowSystem::calculateSurfaceFps(Surface *currentSurface,
                                                  struct timeval *currentTime)
{
    if (currentSurface == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "currentSurface is NULL");
        return;
    }
    if (currentTime == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "currentTime is NULL");
        return;
    }

    if (currentSurface->lastFrameTime.tv_sec == 0
        && currentSurface->lastFrameTime.tv_usec == 0)
    {
        currentSurface->lastFrameTime = *currentTime;
    }

    if (currentTime->tv_sec - currentSurface->lastFrameTime.tv_sec >= FPS_UPDATE_PERIOD)
    {
        float timePassed = timediff(currentSurface->lastFrameTime, *currentTime);
        currentSurface->lastFrameTime = *currentTime;
        currentSurface->surfaceFrameFps = currentSurface->fpsFrameCounter
                                          / timePassed;
        currentSurface->fpsFrameCounter = 0;

        LOG_INFO("WaylandBaseWindowSystem",
                 "surface ID=" << currentSurface->getID() << ", "
                 "surface=" << currentSurface << ", "
                 "client FPS=" << currentSurface->surfaceFps << ", "
                 "render FPS=" << currentSurface->surfaceFrameFps);
    }


}

void WaylandBaseWindowSystem::calculateFps(uint screenID)
{
    struct timeval currentTime;

    // we have rendered a frame
    // calculate & print fps
    gettimeofday(&currentTime, NULL);
    struct screen_data *screenData;
    std::map<uint, struct screen_data*>::iterator match = m_screenData.find(screenID);
    if (match == m_screenData.end())
    {
        screenData = new struct screen_data();
        m_screenData.insert(std::pair<uint, struct screen_data*>(screenID, screenData));
    }
    else
    {
        screenData = match->second;
    }

    if (screenData == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "screenData is NULL");
        return;
    }

    if (screenData->lastTime.tv_sec == 0 && screenData->lastTime.tv_usec == 0)
        screenData->lastTime = currentTime;

    if (currentTime.tv_sec - screenData->lastTime.tv_sec >= FPS_UPDATE_PERIOD)
    {
        float timePassed = timediff(screenData->lastTime, currentTime);
        screenData->lastTime = currentTime;
        float fps = screenData->renderCounter / timePassed;
        screenData->fps = fps;
        screenData->renderCounter = 0;
        LOG_INFO("WaylandBaseWindowSystem",
                 "Screen ID=" << screenID << ", fps=" << fps);
    }

    std::list<Layer*> layers = m_pScene->getCurrentRenderOrder(screenID);
    for (std::list<Layer*>::const_iterator current = layers.begin(); current != layers.end(); current++)
    {
        SurfaceList surfaceList = (*current)->getAllSurfaces();

        for (SurfaceListIterator surfaceIter = surfaceList.begin();
             surfaceIter != surfaceList.end();
             surfaceIter++)
        {
            calculateSurfaceFps((*surfaceIter), &currentTime);
        }
    }
}

void WaylandBaseWindowSystem::RedrawScreen(bool clear, bool swap, uint screen_id)
{
    m_pScene->lockScene();
    graphicSystem->switchScreen(screen_id);
    LayerList layers = m_pScene->getCurrentRenderOrder(screen_id);
    LayerList swLayers;
    // TODO: bRedraw is overly conservative if layers includes a hardware layer
    bool bRedraw = m_forceComposition || graphicSystem->needsRedraw(layers) || (m_systemState == REDRAW_STATE);

    if (bRedraw)
    {
        graphicSystem->activateGraphicContext();
#ifndef WL_OMIT_CLEAR_GB
        if (clear)
        {
            graphicSystem->clearBackground();
        }
#else
        clear = clear;
#endif /* WL_OMIT_CLEAR_GB */
    }
    for(std::list<Layer*>::const_iterator current = layers.begin(); current != layers.end(); current++)
    {
        if ((*current)->getLayerType() == Hardware)
        {
            if (m_forceComposition || graphicSystem->needsRedraw(*current))
            {
                renderHWLayer(*current);
            }
        }
        else if (bRedraw)
        {
            swLayers.push_back(*current);
        }
    }
    if (bRedraw)
    {
#ifdef WL_LOG_DETAIL_TIMER
        struct timeval tv_s;
        struct timeval tv_e;
        float timeSinceLastCalc = 0.0;
        //glFinish();
        gettimeofday(&tv_s, NULL);
        graphicSystem->renderSWLayers(swLayers, false); // Already cleared
        if (swap)
        {
            graphicSystem->swapBuffers();
        }
        //glFinish();
        gettimeofday(&tv_e, NULL);
        timeSinceLastCalc = (float)(tv_e.tv_sec-tv_s.tv_sec) + 0.000001*((float)(tv_e.tv_usec-tv_s.tv_usec));
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "timeSinceLastCalc=" << timeSinceLastCalc << ", "
                  "screen ID=" << screen_id);
#else
        graphicSystem->renderSWLayers(swLayers, false); // Already cleared
        if (swap)
        {
            graphicSystem->swapBuffers();
        }
#endif
        graphicSystem->releaseGraphicContext();

        if (m_debugMode)
        {
            printDebug();
        }
        calculateFps();
        /*we do not reset the system state to IDLE
         * because not all layers are redrawn, only layers on particular screen*/
    }
    m_pScene->unlockScene();
    gettimeofday(&tv0, NULL);
    // update the frame timer
    if (true == m_bUseFrameTimer)
    {
        wl_event_source_timer_update(m_redrawCheckTimer, 10);
    }
}

void WaylandBaseWindowSystem::RedrawAllLayers(bool clear, bool swap)
{
    LayerList layers = m_pScene->getCurrentRenderOrder(0);
    LayerList swLayers;
    // TODO: bRedraw is overly conservative if layers includes a hardware layer
    bool bRedraw = m_forceComposition || graphicSystem->needsRedraw(layers) || (m_systemState == REDRAW_STATE);

    /*check the synchronized surface separately to overrule all redraw reasons*/
    if (graphicSystem->synchronizedSurfacesDamaged(layers) == false)
    {
        bRedraw = false;
    }

    if (bRedraw)
    {
        graphicSystem->activateGraphicContext();
#ifndef WL_OMIT_CLEAR_GB
        if (clear)
        {
            graphicSystem->clearBackground();
        }
#else
        clear = clear;
#endif /* WL_OMIT_CLEAR_GB */
    }
    for (std::list<Layer*>::const_iterator current = layers.begin(); current != layers.end(); ++current)
    {
        if ((*current)->getLayerType() == Hardware)
        {
            if (m_forceComposition || graphicSystem->needsRedraw(*current))
            {
                renderHWLayer(*current);
            }
        }
        else if (bRedraw)
        {
            swLayers.push_back(*current);
        }
    }
    if (bRedraw)
    {
#ifdef WL_LOG_DETAIL_TIMER
        struct timeval tv_s;
        struct timeval tv_e;
        float timeSinceLastCalc = 0.0;
        //glFinish();
        gettimeofday(&tv_s, NULL);
        graphicSystem->renderSWLayers(swLayers, false); // Already cleared
        if (swap)
        {
            graphicSystem->swapBuffers();
        }
        //glFinish();
        gettimeofday(&tv_e, NULL);
        timeSinceLastCalc = (float)(tv_e.tv_sec-tv_s.tv_sec) + 0.000001*((float)(tv_e.tv_usec-tv_s.tv_usec));
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "timeSinceLastCalc=" << timeSinceLastCalc);
#else
        graphicSystem->renderSWLayers(swLayers, false); // Already cleared
        if (swap)
        {
            graphicSystem->swapBuffers();
        }
#endif
        graphicSystem->releaseGraphicContext();

        if (m_debugMode)
        {
            printDebug();
        }

        ClearDamage();

        std::map<uint, struct screen_data*>::iterator match =
            m_screenData.find(0);
        struct screen_data *screenData;
        if (match == m_screenData.end())
        {
            LOG_WARNING("WaylandBaseWindowSystem", "No screenData match found for screen 0");
            screenData = new struct screen_data();
            m_screenData.insert(std::pair<uint, struct screen_data*>(0, screenData));
        }
        else
        {
            screenData = match->second;
        }
        screenData->renderCounter++;

        calculateFps();

        m_systemState = IDLE_STATE;
    }

    std::map<uint, struct repaint_data*>::iterator match;
    struct repaint_data* repaintData;

    match = m_repaintData.begin();
    if (match != m_repaintData.end()) {
        repaintData = match->second;
        repaintData->bRedrawDone = bRedraw;
        repaintData->bRepaintNeeded = false;
    }
    else {
        LOG_ERROR("WaylandBaseWindowSystem", "repaintData is NULL");
    }

    gettimeofday(&tv0, NULL);

    // update the frame timer
    if (true == m_bUseFrameTimer)
    {
        wl_event_source_timer_update(m_redrawCheckTimer, 10);
    }
}

void WaylandBaseWindowSystem::renderHWLayer(Layer *layer)
{
    (void)layer;
}

void WaylandBaseWindowSystem::Redraw()
{
    gettimeofday(&tv0_forRender, NULL);

    // draw all the layers
    //graphicSystem->clearBackground();
    /*LOG_INFO("WaylandBaseWindowSystem","Locking List");*/
    m_pScene->lockScene();

    RedrawAllLayers(true, true); // Clear and Swap

    m_pScene->unlockScene();

    m_forceComposition = false;
}

DumpOutputType WaylandBaseWindowSystem::getDumpType(const std::string& file)
{
    LOG_DEBUG("WaylandBaseWindowSystem","Called, file name=" << file);
    DumpOutputType type = DUMP_UNKNOWN;
    const string bmp_suffix(".bmp");
    if ((file.length() > bmp_suffix.length()) // filename is long enough
        && (file.compare(file.length() - bmp_suffix.length(),
                         bmp_suffix.length(),
                         bmp_suffix.c_str()) == 0))
    {
        type = DUMP_BMP;
    }
    else
    {
        type = DUMP_RAW;
    }

    return type;
}

void WaylandBaseWindowSystem::Screenshot()
{
    LOG_DEBUG("WaylandBaseWindowSystem","Called");
    /*LOG_INFO("WaylandBaseWindowSystem","Locking List");*/
    if (m_pScene == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Cannot take screenshot, m_pScene is NULL");
        return;
    }
    if (graphicSystem == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Cannot take screenshot, graphicSystem is NULL");
        return;
    }
    m_pScene->lockScene();
    graphicSystem->activateGraphicContext();

    DumpOutputType dumpType = getDumpType(m_screenShotFile);

    if (m_takeScreenshot == ScreenshotOfDisplay)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Taking screenshot of screen, ID=" << m_screenShotScreenID);
        // We don't need to redraw anything here, so we can just use the
        // current buffer from the actual screen, no need to switch it to the
        // screenshot output
        graphicSystem->switchScreen(m_screenShotScreenID);
        graphicSystem->saveScreenShotOfFramebuffer(m_screenShotFile, dumpType);
    }
    else if (m_takeScreenshot == ScreenshotOfLayer)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Taking screenshot of layer, ID=" << m_screenShotLayerID << ", "
                  "screen ID=" << m_screenShotScreenID);
        Layer* layer = m_pScene->getLayer(m_screenShotLayerID);

        if (graphicSystem->switchScreen(layer->getContainingScreenId(), true))
        {
            graphicSystem->clearBackground();
            if (layer != NULL)
            {
                graphicSystem->renderSWLayer(layer, false); // Do clear
            }
            if (dumpType == DUMP_BMP)
            {
                graphicSystem->saveScreenShotOfFramebuffer(m_screenShotFile, dumpType);
            }
            else if (dumpType == DUMP_RAW)
            {
                graphicSystem->saveScreenShotOfLayer(m_screenShotFile, layer);
            }
            else
            {
                LOG_WARNING("WaylandBaseWindowSystem",
                            "Unexpected dump type, value=" << dumpType);
            }
        }
        else
        {
            LOG_WARNING("WaylandBaseWindowSystem",
                        "Failed to switch to screenshot output");
        }
    }
    else if (m_takeScreenshot == ScreenshotOfSurface)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Taking screenshot of surface, ID=" << m_screenShotSurfaceID
                  << " on layer, ID=" << m_screenShotLayerID);
        Layer* layer = m_pScene->getLayer(m_screenShotLayerID);

        if (layer == NULL)
        {
            LOG_WARNING("WaylandBaseWindowSystem",
                        "Layer not retrieved for layer, ID="
                        << m_screenShotLayerID);
        }
        else
        {
            Surface* surface = m_pScene->getSurface(m_screenShotSurfaceID);

            if (surface == NULL)
            {
                LOG_WARNING("WaylandBaseWindowSystem",
                            "Surface not retrieved for surface, ID="
                            << m_screenShotSurfaceID);
            }
            else
            {
                bool isRGB;
                if (!graphicSystem->getSurfaceIsRGB(surface, isRGB))
                {
                    LOG_WARNING("WaylandBaseWindowSystem",
                                "isRGB not known for surface, ID="
                                << surface->getID());
                }
                else
                {
                    if (graphicSystem->switchScreen(layer->getContainingScreenId(), true))
                    {
                        if (dumpType == DUMP_BMP)
                        {
                            graphicSystem->clearBackground();
                            graphicSystem->beginLayer(layer);
                            graphicSystem->renderSurface(surface);
                            graphicSystem->endLayer();

                            graphicSystem->saveScreenShotOfFramebuffer(m_screenShotFile,
                                                                       dumpType);
                        }
                        else if (dumpType == DUMP_RAW)
                        {
                            if (isRGB)
                            {
                                graphicSystem->dumpSurfaceToFile(m_screenShotFile,
                                                                 surface);
                            }
                            else
                            {
                                graphicSystem->clearBackground();
                                graphicSystem->beginLayer(layer);
                                graphicSystem->renderSurface(surface);
                                graphicSystem->endLayer();

                                graphicSystem->saveScreenShotOfSurface(m_screenShotFile,
                                                                       surface);
                            }
                        }
                        else
                        {
                            LOG_WARNING("WaylandBaseWindowSystem",
                                        "Unexpected dump type, code=" << dumpType);
                        }
                    }
                    else
                    {
                        LOG_WARNING("WaylandBaseWindowSystem",
                                    "Failed to switch to screenshot output");
                    }
                } // known isRGB
            } // surface != NULL
        } // layer != NULL
    } // takeScreenshot == ScreenshotOfSurface


    m_takeScreenshot = ScreenShotNone;
    LOG_DEBUG("WaylandBaseWindowSystem", "Done taking screenshot");

    graphicSystem->releaseGraphicContext();
    m_pScene->unlockScene();
    /*LOG_INFO("WaylandBaseWindowSystem","UnLocking List");*/
}

void WaylandBaseWindowSystem::destroyListenerSurfaceBuffer(struct wl_listener* listener, void *data)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "listener=" << listener << ", data=" << data);
    struct native_surface *es = wl_container_of(listener, es, buffer_destroy_listener);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)es->windowSystem);

    windowSystem->m_pScene->lockScene();

    Surface* ilmSurface = windowSystem->getSurfaceFromNativeSurface(es);

    if (NULL != ilmSurface)
    {
         LOG_DEBUG("WaylandBaseWindowSystem",
                   "Destroying surface, ID=" << ilmSurface->getID() << ", "
                   "containing layer ID="
                   << ilmSurface->getContainingLayerId() << ", "
                   "native surface=" << es);

         windowSystem->removeClientBuffer(ilmSurface);
    }

    es->buffer.buffer = NULL;
    es->buffer.busy_count = 0;

    windowSystem->m_pScene->unlockScene();
}

void WaylandBaseWindowSystem::destroyListenerSurfacePendingBuffer(struct wl_listener* listener, void *data)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "listener=" << listener << ", data=" << data);
    struct native_surface *es = wl_container_of(listener, es, pending.buffer_destroy_listener);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)es->windowSystem);

    windowSystem->m_pScene->lockScene();

    Surface* ilmSurface = windowSystem->getSurfaceFromNativeSurface(es);

    if (NULL != ilmSurface)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Destroying surface, ID=" << ilmSurface->getID() << ", "
                  "containing layer ID="
                  << ilmSurface->getContainingLayerId() << ", "
                  "native surface=" << es);

        windowSystem->removeClientBuffer(ilmSurface);
    }

    es->pending.buffer.buffer = NULL;
    es->pending.buffer.busy_count = 0;

    windowSystem->m_pScene->unlockScene();
}

struct native_surface* WaylandBaseWindowSystem::createNativeSurface()
{
    struct native_surface* surface;
    surface = (struct native_surface*)calloc(1, sizeof *surface);
    if (NULL == surface)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Failed to create native surface");
        return NULL;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Created native surface=" << surface << ", "
                  "window system=" << this);
    }

    wl_signal_init(&surface->surface.resource.destroy_signal);

    wl_list_init(&surface->link);

    surface->surface.resource.client = NULL;

    surface->windowSystem = this;
    // TODO visual
    // surface->visual = NONE_VISUAL;

    surface->buffer.buffer = NULL;
    surface->buffer.busy_count = 0;
    surface->buffer_destroy_listener.notify = destroyListenerSurfaceBuffer;
    surface->pending.buffer_destroy_listener.notify = destroyListenerSurfacePendingBuffer;
    wl_list_init(&surface->pending.frame_callback_list);

    return surface;
}

uint32_t WaylandBaseWindowSystem::getTime(void)
{
#ifdef WL_OMIT_GETTIME
    return 0xF0F0F0F0;
#else /* WL_OMIT_GETTIME */
    struct timeval tv;

    gettimeofday(&tv, NULL);

    return tv.tv_sec * 1000 + tv.tv_usec / 1000;
#endif /* WL_OMIT_GETTIME */
}

void WaylandBaseWindowSystem::removeClientBuffer(Surface* ilmSurface)
{
    if ((NULL != ilmSurface))
    {
        this->graphicSystem->activateGraphicContext();
        this->graphicSystem->getTextureBinder()->destroyClientBuffer(ilmSurface);
        this->graphicSystem->releaseGraphicContext();
    }
    else
    {
        LOG_ERROR("WaylandBaseWindowSystem",
              "ilmSurface is NULL" );
    }
    return;
}

void WaylandBaseWindowSystem::destroySurfaceCallback(struct wl_resource* resource)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, resource=" << resource);
    struct native_surface* nativeSurface = wl_container_of(resource, nativeSurface, surface.resource);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);

    windowSystem->m_pScene->lockScene();

    Surface* ilmSurface = windowSystem->getSurfaceFromNativeSurface(nativeSurface);
    struct native_frame_callback* cb = NULL;
    struct native_frame_callback* cb_installed = NULL;
    struct native_frame_callback* next = NULL;
    struct native_frame_callback* next_installed = NULL;
    WlInputEventList *pWlIpEvent = windowSystem->getInputEventList();
    WlInputEventList::iterator it;

    if (pWlIpEvent)
    {
        /* Reset the focus for all seats in case if this surface
         * has any focus */
        for (it = pWlIpEvent->begin(); it != pWlIpEvent->end(); it++)
        {
            (*it)->inputDevice().ResetAllFocuses(&nativeSurface->surface);
        }
    }
    else if (windowSystem->m_inputEvent)
    {
        /* window system is not supporting multiple seats. Hence
         * reset focus for default seat (seat0)
         */
        windowSystem->m_inputEvent->inputDevice().ResetAllFocuses(&nativeSurface->surface);
    }

    if (NULL != ilmSurface)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Destroying surface, ID=" << ilmSurface->getID() << ", "
                  "containing layer ID="
                  << ilmSurface->getContainingLayerId() << ", "
                  "native surface=" << nativeSurface << ", "
                  "resource=" << resource);

        windowSystem->removeClientBuffer(ilmSurface);
        ilmSurface->removeNativeContent();
        ilmSurface->setNativeBuffer(0);
        if (true == ilmSurface->ackClearDamage)
        {
            windowSystem->m_pScene->sigSurfaceDamageClear();
            ilmSurface->ackClearDamage = false;
        }

        if (NULL != ilmSurface->platform)
        {
            delete ilmSurface->platform;
            ilmSurface->platform = NULL;
        }
    }

    wl_list_for_each_safe(cb, next, &nativeSurface->pending.frame_callback_list, link)
    {
        for (std::map<uint, struct repaint_data*>::iterator it = windowSystem->m_repaintData.begin();
             it != windowSystem->m_repaintData.end();
             it++)
        {
            struct repaint_data* repaintData = it->second;

            /*remove all already installed frame callback of current surface to delete*/
            wl_list_for_each_safe(cb_installed, next_installed, &repaintData->listFrameCallback, link)
            {
                if (cb == cb_installed)
                {
                    wl_list_remove(&cb_installed->link);
                }
            }

            cb_installed = NULL;
            next_installed = NULL;

            wl_list_for_each_safe(cb_installed, next_installed, &windowSystem->m_listFrameCallback, link)
            {
                if (cb == cb_installed)
                {
                    wl_list_remove(&cb_installed->link);
                }
            }
        }
        wl_resource_destroy(&cb->resource);
    }

    wl_list_remove(&nativeSurface->link);

    if (nativeSurface->pending.buffer.buffer)
    {
        wl_list_remove(&nativeSurface->pending.buffer_destroy_listener.link);
    }

    if (nativeSurface->pending.buffer.buffer)
    {
        struct lm_wl_buffer* buffer = NULL;
        buffer = &(nativeSurface->pending.buffer);
        if (nativeSurface->buffer.buffer)
        {
            windowSystem->postReleaseBuffer(&nativeSurface->buffer);
            wl_list_remove(&nativeSurface->buffer_destroy_listener.link);
        }
        if (buffer->buffer)
        {
            wl_signal_emit(&buffer->resource->destroy_signal, &nativeSurface->buffer_destroy_listener);
        }
    }

    free(nativeSurface);

    windowSystem->m_pScene->unlockScene();
}

void WaylandBaseWindowSystem::destroyClientBufferShm(Surface* ilmSurface)
{

    removeClientBuffer(ilmSurface);

    return;
}

extern "C" void WaylandBaseWindowSystem::surfaceIFDestroy(struct wl_client *client, struct wl_resource *resource)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "resource=" << resource);
    wl_resource_destroy(resource);
}

void WaylandBaseWindowSystem::postReleaseBuffer(struct lm_wl_buffer *buffer)
{
    if (--buffer->busy_count > 0)
    {
        return;
    }

    if (NULL == buffer->resource->client)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Release client is NULL, buffer=" << buffer);
    }
    wl_resource_post_event(buffer->resource, WL_BUFFER_RELEASE);
}

void WaylandBaseWindowSystem::attachBufferToNativeSurface(struct lm_wl_buffer* buffer, struct wl_surface* surface)
{

    struct native_surface* nativeSurface = (struct native_surface*)surface;
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);
    Surface* ilmSurface = windowSystem->getSurfaceFromNativeSurface(nativeSurface);
    if (NULL == ilmSurface)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Failed to get surface for wl_surface, "
                    "native surface=" << surface);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Got surface, buffer=" << buffer << ", "
                  "native surface=" << nativeSurface << ", "
                  "native surface ID=" << nativeSurface->surface.resource.object.id);
    }

    ilmSurface->setNativeBuffer(static_cast<void *>(buffer));
    WaylandPlatformSurface* nativePlatformSurface = (WaylandPlatformSurface*)ilmSurface->platform;
    if (0 != nativePlatformSurface)
    {
        windowSystem->graphicSystem->activateGraphicContext();
        windowSystem->graphicSystem->getTextureBinder()->createClientBuffer(ilmSurface);
        windowSystem->graphicSystem->releaseGraphicContext();
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "nativePlatformSurface->enable, "
                  "native surface=" << surface << ", "
                  "wayland platform surface=" << nativePlatformSurface);
        nativePlatformSurface->enableRendering();
    }
}

extern "C" void WaylandBaseWindowSystem::surfaceIFAttach(struct wl_client* client,
            struct wl_resource* resource,
            struct wl_resource* buffer_resource, int32_t x, int32_t y)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "resource=" << resource << ", "
              "buffer_resource=" << buffer_resource << ", "
              "x=" << x << ", y=" << y);

    struct native_surface* nativeSurface = (struct native_surface*)resource->data;
    struct wl_buffer* buffer = NULL;
    WaylandBaseWindowSystem* windowSystem;

    if (buffer_resource)
    {
        buffer = (struct wl_buffer*)buffer_resource->data;
        if (buffer == NULL)
        {
            wl_client_post_no_memory(client);
            return;
        }
    }

     windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);

     if(windowSystem)
     {
         windowSystem->m_pScene->lockScene();
     }


    if (nativeSurface->pending.buffer.buffer)
    {
        wl_list_remove(&nativeSurface->pending.buffer_destroy_listener.link);
    }

    nativeSurface->pending.sx = x;
    nativeSurface->pending.sy = y;

    nativeSurface->pending.buffer.resource = buffer_resource;

    nativeSurface->pending.buffer.busy_count = 0;

    if (nativeSurface->pending.buffer.resource)
    {
        wl_signal_add(&(nativeSurface->pending.buffer.resource->destroy_signal),
                      &nativeSurface->pending.buffer_destroy_listener);
        nativeSurface->pending.remove_contents = 0;
    }
    else
    {
        nativeSurface->pending.remove_contents = 1;
    }

     if(windowSystem)
    {
        windowSystem->m_pScene->unlockScene();
    }
}

extern "C" void WaylandBaseWindowSystem::surfaceIFDamage(struct wl_client *client,
            struct wl_resource *resource,
            int32_t x, int32_t y, int32_t width, int32_t height)
{
    struct native_surface* nativeSurface = (struct native_surface*)resource->data;
    WaylandBaseWindowSystem* windowSystem;

    if (NULL == nativeSurface)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Invalid native surface, "
                    "client=" << client << ", "
                    "resource=" << resource << ", "
                    "x=" << x << ", y=" << y << ", width=" << width << ", "
                    "height=" << height);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Set native surface, client=" << client << ", "
                  "resource=" << resource << ", "
                  "x=" << x << ", y=" << y << ", width=" << width << ", "
                  "height=" << height << ", "
                  "native surface=" << nativeSurface);
    }

   windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);

    if(windowSystem)
    {
        windowSystem->m_pScene->lockScene();
    }

    nativeSurface->pending.damaged = true;
    nativeSurface->pending.damage.x = x;
    nativeSurface->pending.damage.y = y;
    nativeSurface->pending.damage.width = width;
    nativeSurface->pending.damage.height = height;

    if(windowSystem)
    {
        windowSystem->m_pScene->unlockScene();
    }
}

extern "C" void WaylandBaseWindowSystem::destroyFrameCallback(struct wl_resource *resource)
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called, resource=" << resource);
    struct native_frame_callback* cb = (struct native_frame_callback*)resource->data;

    wl_list_remove(&cb->link);
    free(cb);
    // Not clearing resource->data here, because resource is always a member of cb that was
    // freed before.
}

extern "C" void WaylandBaseWindowSystem::surfaceIFFrame(struct wl_client *client,
            struct wl_resource *resource, uint32_t callback)
{
    struct native_frame_callback* cb;
    struct native_surface* es = (struct native_surface*)resource->data;

    cb = (struct native_frame_callback*)malloc(sizeof *cb);
    if (NULL == cb)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Unable to create native frame callback, "
                    "client=" << client << ", "
                    "resource=" << resource << ", callback=" << callback);
        wl_resource_post_no_memory(resource);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Creating native frame callback, "
                  "client=" << client << ", "
                  "resource=" << resource << ", callback=" << callback << ", "
                  "native frame callback=" << cb);
    }

    cb->resource.object.interface = &wl_callback_interface;
    cb->resource.object.id = callback;
    cb->resource.destroy = destroyFrameCallback;
    cb->resource.client = client;
    cb->resource.data = cb;

    wl_client_add_resource(client, &cb->resource);
    wl_list_insert(es->pending.frame_callback_list.prev, &cb->link);
}

extern "C" void WaylandBaseWindowSystem::surfaceIFSetOpaqueRegion(struct wl_client *client,
        struct wl_resource *resource, struct wl_resource *region_resource)
{
    struct native_surface* nativeSurface = (struct native_surface*)resource->data;
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);

    if(NULL == windowSystem)
    {
        LOG_ERROR("WaylandBaseWindowSystem",
                   "surfaceIFSetOpaqueRegion-windowsystem is NULL nativeSurface="<<nativeSurface);
        return;
    }

    Surface* surface = windowSystem->getSurfaceFromNativeSurface(nativeSurface);

    LOG_INFO("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "resource=" << resource << ", "
              "native surface=" << nativeSurface << ", "
              "surface=" << surface);

    if (NULL != surface)
    {
        surface->m_opaqueRegion = true;
        LOG_DEBUG("WaylandBaseWindowSystem",
                "Surface opaque region set, client=" << client << ", "
		        "resource=" << resource << ", "
		        "native surface=" << nativeSurface << ", "
		        "surface ID=" << surface->getID());
    }
    else
    {
        LOG_WARNING("WaylandBaseWindowSystem", "surface is NULL for native surface=" << nativeSurface);
    }
}

extern "C" void WaylandBaseWindowSystem::surfaceIFSetInputRegion(struct wl_client *client,
        struct wl_resource *resource, struct wl_resource *region_resource)
{
    LOG_INFO("WaylandBaseWindowSystem",
               "set_input_region request is not supported ");
    (void)client;
    (void)resource;
    (void)region_resource;
}

extern "C" void WaylandBaseWindowSystem::surfaceIFCommit(struct wl_client *client, struct wl_resource *resource)
{
    struct native_surface* nativeSurface = (struct native_surface*)resource->data;
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)nativeSurface->windowSystem);

     if(NULL == windowSystem)
     {
         LOG_ERROR("WaylandBaseWindowSystem",
                    "surfaceIFCommit-windowsystem is NULL nativeSurface="<<nativeSurface);
         return;
     }

     /* Locking scene mutex */
    windowSystem->m_pScene->lockScene();

    /* surface_frame process */
    windowSystem->checkForNewSurfaceNativeContent();

    Surface* surface = windowSystem->getSurfaceFromNativeSurface(nativeSurface);
    struct lm_wl_buffer* buffer = NULL;

    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "resource=" << resource << ", "
              "native surface=" << nativeSurface << ", "
              "surface=" << surface);


    /* wl_surface_attach process */
    if (nativeSurface->pending.buffer.buffer || nativeSurface->pending.remove_contents)
    {
        buffer = &(nativeSurface->pending.buffer);
        if (nativeSurface->buffer.buffer)
        {
            windowSystem->postReleaseBuffer(&nativeSurface->buffer);
            wl_list_remove(&nativeSurface->buffer_destroy_listener.link);
        }
        if (buffer->buffer)
        {
            buffer->busy_count++;
            wl_signal_add(&(buffer->resource->destroy_signal), &nativeSurface->buffer_destroy_listener);
        }
    }

    /* Increment update counters */
    struct timeval currentTime;
    gettimeofday(&currentTime, NULL);

    if (NULL != surface)
    {
        bool srcORdstRegionChanged = surface->m_deferredSrcRegionSet ||
                surface->m_deferredDstRegionSet;
        if (surface->m_deferredSrcRegionSet)
        {
            if (surface->setSourceRegion(surface->getSourceRegionDeferred()))
            {
                //TODO: send notification on change of source region
            }
    	    surface->m_deferredSrcRegionSet = false;
        }
        if (surface->m_deferredDstRegionSet)
        {
            if(surface->setDestinationRegion(surface->getDestinationRegionDeferred()))
            {
                //TODO: send notification on change of source region
            }
        	surface->m_deferredDstRegionSet = false;
        }
        if (srcORdstRegionChanged)
        {
            unsigned int layerid = surface->getContainingLayerId();
            if (layerid != Layer::INVALID_ID)
            {
                Layer* l = windowSystem->m_pScene->getLayer(layerid);
                surface->calculateTargetDestination(l->getSourceRegion(),l->getDestinationRegion());
            }
        }

        surface->updateCounter++;
        surface->fpsUpdateCounter++;
        if (surface->lastUpdateTime.tv_sec == 0 && surface->lastUpdateTime.tv_usec == 0)
        {
            surface->lastUpdateTime = currentTime;
        }
        
        if (currentTime.tv_sec - surface->lastUpdateTime.tv_sec >= FPS_UPDATE_PERIOD)
        {
            float timePassed = timediff(surface->lastUpdateTime, currentTime);
            surface->lastUpdateTime = currentTime;
            surface->surfaceFps = surface->fpsUpdateCounter / timePassed;
            surface->fpsUpdateCounter = 0;
        }
        /* surface_damage process */
        if (nativeSurface->pending.damaged == true)
        {
            surface->damaged = true;
        }
        else
        {
            surface->damaged = false;
        }

        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Surface damage state set, client=" << client << ", "
                  "resource=" << resource << ", "
                  "native surface=" << nativeSurface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "surface damage=" << surface->damaged);
    }
    else
    {
        LOG_WARNING("WaylandBaseWindowSystem", "surface is NULL for native surface=" << nativeSurface);
    }

    windowSystem->attachBufferToNativeSurface(buffer, &nativeSurface->surface);

    if (buffer)
    {
        /* move pending buffer to be main buffer */
        nativeSurface->buffer.buffer = buffer->buffer;
        nativeSurface->buffer.busy_count = buffer->busy_count;
    }
    else
    {
        /* remove main buffer */
        nativeSurface->buffer.buffer = NULL;
        nativeSurface->buffer.busy_count = 0;
    }

    windowSystem->scheduleRepaintForSurface(nativeSurface);

    /* Unlocking scene mutex */
    windowSystem->m_pScene->unlockScene();
}

extern "C" void WaylandBaseWindowSystem::surfaceIFSetBufferTransform(struct wl_client *client,
        struct wl_resource *resource, int32_t transform)
{
    LOG_WARNING("WaylandBaseWindowSystem",
               "set_buffer_transform request is not supported ");
    (void)client;
    (void)resource;
    (void)transform;
}

extern "C" void WaylandBaseWindowSystem::surfaceIFSetBufferScale(struct wl_client *client,
        struct wl_resource *resource, int32_t scale)
{
    LOG_WARNING("WaylandBaseWindowSystem",
               "set_buffer_scale request is not supported ");
    (void)client;
    (void)resource;
    (void)scale;
}

extern "C" void WaylandBaseWindowSystem::surfaceIFDamageBuffer(struct wl_client *client,
          struct wl_resource *resource, int32_t x, int32_t y, int32_t width, int32_t height)
{
    LOG_WARNING("WaylandBaseWindowSystem",
               "set_damage_buffer request is not supported ");
    (void)client;
    (void)resource;
    (void)x;
    (void)y;
    (void)width;
    (void)height;
}

extern "C" const struct wl_surface_interface g_surfaceInterface = {
    WaylandBaseWindowSystem::surfaceIFDestroy,
    WaylandBaseWindowSystem::surfaceIFAttach,
    WaylandBaseWindowSystem::surfaceIFDamage,
    WaylandBaseWindowSystem::surfaceIFFrame,
    WaylandBaseWindowSystem::surfaceIFSetOpaqueRegion,
    WaylandBaseWindowSystem::surfaceIFSetInputRegion,
    WaylandBaseWindowSystem::surfaceIFCommit,
    WaylandBaseWindowSystem::surfaceIFSetBufferTransform,
    WaylandBaseWindowSystem::surfaceIFSetBufferScale,
    WaylandBaseWindowSystem::surfaceIFDamageBuffer
};

void WaylandBaseWindowSystem::destroyRegionCallback(struct wl_resource* resource)
{
    struct compositor_region *region;

    region = (struct compositor_region *)wl_resource_get_user_data(resource);
    free(region);
}

extern "C" void WaylandBaseWindowSystem::regionIFDestroy
        (struct wl_client *client, struct wl_resource *resource)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, client=" << client << ", "
              "resource=" << resource);
    wl_resource_destroy(resource);
}
extern "C" void WaylandBaseWindowSystem::regionIFAdd
        (struct wl_client *client, struct wl_resource *resource,
        int32_t x, int32_t y, int32_t width, int32_t height)
{
	LOG_INFO("WaylandBaseWindowSystem",
               "wl_region_interface is not supported ");
    (void)client;
    (void)resource;
    (void)x;
    (void)y;
    (void)width;
    (void)height;
}
extern "C" void WaylandBaseWindowSystem::regionIFSubstract
        (struct wl_client *client, struct wl_resource* resource,
        int32_t x, int32_t y, int32_t width, int32_t height)
{
	LOG_INFO("WaylandBaseWindowSystem",
               "wl_region_interface is not supported ");
    (void)client;
    (void)resource;
    (void)x;
    (void)y;
    (void)width;
    (void)height;
}

extern "C" const struct wl_region_interface g_regionInterface = {
    WaylandBaseWindowSystem::regionIFDestroy,
    WaylandBaseWindowSystem::regionIFAdd,
    WaylandBaseWindowSystem::regionIFSubstract
};

extern "C" void WaylandBaseWindowSystem::compositorIFCreateRegion
        (struct wl_client *client, struct wl_resource* resource, uint32_t id)
{
    struct compositor_region *region;

    region = (struct compositor_region *)malloc(sizeof *region);
    if (region == NULL) {
        wl_resource_post_no_memory(resource);
        return;
    }

    region->resource =
        wl_resource_create(client, &wl_region_interface, 1, id);
    if (region->resource == NULL) {
        free(region);
        wl_resource_post_no_memory(resource);
        return;
    }
    wl_resource_set_implementation(region->resource, &g_regionInterface,
                                   region, destroyRegionCallback);
}

extern "C" void WaylandBaseWindowSystem::compositorIFCreateSurface
        (struct wl_client *client, struct wl_resource* resource, uint32_t id)
{
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*) resource->data);
    struct native_surface* surface;

     if(NULL == windowSystem)
     {
         LOG_ERROR("WaylandBaseWindowSystem",
                    "compositorIFCreateSurface-windowsystem is NULL ");
         return;
     }

    windowSystem->m_pScene->lockScene();

    surface = windowSystem->createNativeSurface();
    if (NULL == surface)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Unable to create native surface, "
                    "client=" << client << ", "
                    "resource=" << resource << ", "
                    "native surface ID=" << id << ", "
                    "native surface=" << surface);
        wl_resource_post_no_memory(resource);
       windowSystem->m_pScene->unlockScene();
        return;
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Called, created native surface, "
                  "client=" << client << ", "
                  "resource=" << resource << ", "
                  "native surface ID=" << id << ", "
                  "native surface=" << surface);
    }

    surface->surface.resource.destroy = destroySurfaceCallback;
    surface->windowSystem = windowSystem;

    surface->surface.resource.object.id = id;
    surface->surface.resource.object.interface = &wl_surface_interface;
    surface->surface.resource.object.implementation = (void (**)(void))&g_surfaceInterface;
    surface->surface.resource.data = surface;

    struct serverinfoClient* serverinfoPairNode;

    wl_list_for_each(serverinfoPairNode, &windowSystem->m_connectionList, link)
    {
        if (serverinfoPairNode->client != client)
        {
            continue;
        }
        surface->connectionId = serverinfoPairNode->connectionId;
        break;
    }

    windowSystem->checkForNewSurfaceNativeContent();
    wl_client_add_resource(client, &surface->surface.resource);
    wl_list_insert(windowSystem->m_nativeSurfaceList.prev, &surface->link);

    windowSystem->m_pScene->unlockScene();
}

const static struct wl_compositor_interface g_compositorInterface =
{
    WaylandBaseWindowSystem::compositorIFCreateSurface,
    WaylandBaseWindowSystem::compositorIFCreateRegion
};

/**
 * Thread in charge of the CompositorWindow eventloop
 * Friend function of class WaylandBaseWindowSystem
 */
extern "C" void* WaylandBaseWindowSystem::eventLoopCallback(void *ptr)
{
    WaylandBaseWindowSystem *windowsys = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*) ptr);
    return windowsys->eventLoop();
}

void WaylandBaseWindowSystem::bindCompositor(struct wl_client* client, void* data, uint32_t version, uint32_t id)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "bindCompositor client=" << client << ", "
              "data=" << data << ", version=" << version << ", ID=" << id);
    wl_client_add_object(client, &wl_compositor_interface, &g_compositorInterface, id, data);
}


void WaylandBaseWindowSystem::repaint(int msecs)
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called, msecs=" << msecs);

    Redraw();

    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
}

void WaylandBaseWindowSystem::chkForDrawPending()
{
    /*Add an event check for pending redraw because when
     *page flip is pending we cannot draw*/
    wl_event_source_timer_update(m_redrawCheckTimer, 1);
}

void WaylandBaseWindowSystem::finishFrame()
{
    repaint(getTime());

    WaylandEvdevInputEvent* inputEvent = (WaylandEvdevInputEvent *)m_inputEvent;

    if (inputEvent == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "m_inputEvent is NULL");
    }
    else
    {
        struct wl_event_loop *eventLoop = inputEvent->getEventLoop();
        if (eventLoop == NULL)
        {
            LOG_WARNING("WaylandBaseWindowSystem", "Event loop is NULL");
        }
        else
        {
            wl_event_loop_dispatch(eventLoop, 0);
        }
    }
}

void WaylandBaseWindowSystem::idleEventRepaint(void *data)
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called, data=" << data);
    struct repaint_data* repaintData = static_cast<struct repaint_data*>(data);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(repaintData->baseWindowSystem);

    /* if the timer is already triggered,
     * there is no need to start the repaint loop again
     * by triggering the timer */
    if (windowSystem && (!repaintData->bRepaintTimerTriggered))
    {
        if (!repaintData->bUseRepaintTimer)
            WaylandBaseWindowSystem::repaintTimerHandler((void*)repaintData);
        else
            windowSystem->graphicSystem->startRepaintLoop(repaintData->screenId);
    }
    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
}

void WaylandBaseWindowSystem::updateRepaintTimer(int screenID, const struct timespec *stamp,
        int repaintWindowPeriodMS, uint refreshNS, int flags)
{
    std::map<uint, struct repaint_data*>::iterator match;
    struct repaint_data* repaintData;
    struct timespec current_time;
    struct timespec gone_time;
    int msec;

    LOG_DEBUG("WaylandBaseWindowSystem", "Called, screenID=" << screenID
              << " stamp=" << stamp << " repaintWindowPeriodMS=" << repaintWindowPeriodMS
              << " flags=" << flags);

    match = m_repaintData.find(screenID);

    if(match == m_repaintData.end()) {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "ScreenId " << screenID <<
                    " is not present in m_repaintData map");
        return;
    }
    else {
        repaintData = match->second;
    }

    if (!repaintData->bUseRepaintTimer)
    {
        if (repaintData->bRepaintNeeded)
        {
            WaylandBaseWindowSystem::repaintTimerHandler((void*)repaintData);
        }
        return;
    }

    clock_gettime(CLOCK_MONOTONIC, &current_time);

    LOG_DEBUG("Gal2dGraphicSystem","Current Time: sec="<< current_time.tv_sec << ", " <<
              "nsec=" << current_time.tv_nsec);

    LOG_DEBUG("Gal2dGraphicSystem","Flip Time: sec="<< stamp->tv_sec << ", " <<
              "nsec=" << stamp->tv_nsec);

    timespecSub(&gone_time, &current_time, stamp);

    LOG_DEBUG("Gal2dGraphicSystem","Gone Time: sec="<< gone_time.tv_sec << ", " <<
              "nsec=" << gone_time.tv_nsec << ", " <<
              "ConvNS=" << timespecToNsec(&gone_time));

    LOG_DEBUG("WaylandGal2dWindowSystem",
              "repaintWindowPeriodMS=" << repaintWindowPeriodMS << ", " <<
              "refreshNS=" << refreshNS);

    msec = (refreshNS - timespecToNsec(&gone_time)) / 1000000; /* floor */
    msec -= repaintWindowPeriodMS;

    LOG_DEBUG("Gal2dGraphicSystem","TimerMsec="<< msec);

    if (msec < -1000 || msec > 1000) {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "Computed repaint delay is fatal, "
                    "resetting it to 0");
        msec = 0;
    }

    /*called from repaint_start_loop*/
    if ((flags == REPAINT_TIMER_UPDATE_INVALID) && (msec < 0)) {
        msec += refreshNS / 1000000;
    }

    repaintData->repaintWindowPeriodMS = repaintWindowPeriodMS;

    /* if the msec set to 0 or falls in -ve value
     * call the repaintTimerHandler immediately
     */
    if (msec < 1)
        WaylandBaseWindowSystem::repaintTimerHandler((void*)repaintData);
    else {
        wl_event_source_timer_update(repaintData->handleRepaintTimer, msec);
        repaintData->bRepaintTimerTriggered = true;
    }

    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
}

float WaylandBaseWindowSystem::getScreenFPS(uint screenID)
{
    struct screen_data* screenData;

    std::map<uint, struct screen_data*>::iterator match = m_screenData.find(screenID);
    if (match == m_screenData.end())
    {
        return 0.0f;
    }
    else
    {
        screenData = match->second;
    }

    return screenData->fps;
}

void WaylandBaseWindowSystem::idleEventStartupDone(void *data)
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called, data=" << data);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>(data);
    if (windowSystem)
    {
        sem_post(&windowSystem->sem_run);
    }
}

void WaylandBaseWindowSystem::scheduleRepaint(void *data)
{
    struct repaint_data* repaintData = (struct repaint_data*)data;
    int screenId = -1;

    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, data=" << data);

    if (NULL == repaintData) {
        LOG_ERROR("WaylandBaseWindowSystem",
                  "repaintData is NULL, so repaint not scheduled");
        return;
    }

    LOG_DEBUG("WaylandBaseWindowSystem",
              "repaintData=" << repaintData << ", "
              "screenId=" << repaintData->screenId << ", "
              "bRepaintNeeded=" << repaintData->bRepaintNeeded << ", "
              "bRepaintScheduled=" << repaintData->bRepaintScheduled);

    repaintData->bRepaintNeeded = true;

    if (repaintData->bRepaintScheduled)
        return;

    struct wl_event_loop *loop = wl_display_get_event_loop(m_wlDisplay);
    if (loop)
    {
        wl_event_loop_add_idle(loop, WaylandBaseWindowSystem::idleEventRepaint, repaintData);
        repaintData->bRepaintScheduled = true;
    }
}

bool WaylandBaseWindowSystem::initCompositor()
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called");

    m_wlCompositorGlobal = wl_display_add_global(m_wlDisplay, &wl_compositor_interface, this, bindCompositor);
    if (NULL == m_wlCompositorGlobal)
    {
        LOG_WARNING("WaylandBaseWindowSystem",
                    "wl_display_add_global, "
                    "failed to set wl_compositor_interface, "
                    "wayland display=" << m_wlDisplay);
        return false;
    }
    LOG_DEBUG("WaylandBaseWindowSystem", "wl_display_add_global, SUCCESS");
    wl_display_init_shm(m_wlDisplay);

    wl_list_init(&m_listFrameCallback);
    wl_list_init(&m_connectionList);
    wl_list_init(&m_nativeSurfaceList);
    createServerinfo(this);
    createOutputinfo(this);

    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
    return true;
}

void WaylandBaseWindowSystem::shutdownCompositor()
{
    LOG_DEBUG("WaylandBaseWindowSystem", "Called");

    if (NULL != m_serverInfoGlobal)
    {
        wl_display_remove_global(m_wlDisplay, m_serverInfoGlobal);
        m_serverInfoGlobal = NULL;
    }

    if (NULL != m_serverInfo)
    {
        free(m_serverInfo);
        m_serverInfo = NULL;
    }

    if (NULL != m_wlCompositorGlobal)
    {
        wl_display_remove_global(m_wlDisplay, m_wlCompositorGlobal);
        m_wlCompositorGlobal = NULL;
    }
}

void WaylandBaseWindowSystem::wakeUpRendererThread()
{
}

int WaylandBaseWindowSystem::signalEventOnTerm(int signal_number, void *data)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, caught signal_number=" << signal_number << ", "
              "data=" << data);
    WaylandBaseWindowSystem* windowSystem = static_cast<WaylandBaseWindowSystem*>((WaylandBaseWindowSystem*)data);

    if (windowSystem && windowSystem->getNativeDisplayHandle())
    {
        wl_display_terminate(windowSystem->getNativeDisplayHandle());
    }

    return 1;
}

bool WaylandBaseWindowSystem::isScreenWiseRenderingNeeded()
{
    return false;
}

void WaylandBaseWindowSystem::initRepaintTimer(void)
{
    struct wl_event_loop *loop = wl_display_get_event_loop(m_wlDisplay);

    struct repaint_data* repaintData = new struct repaint_data();

    repaintData->handleRepaintTimer = wl_event_loop_add_timer(loop, repaintTimerHandler, repaintData);
    repaintData->screenId = 0;
    repaintData->baseWindowSystem = (BaseWindowSystem*)this;
    repaintData->bRepaintNeeded = 0;
    repaintData->bRepaintScheduled = 0;
    repaintData->bRepaintTimerTriggered = 0;
    repaintData->bScreenDamaged = 0;
    repaintData->repaintWindowPeriodMS = 0;
    repaintData->repaintCount = 0;
    repaintData->bUseRepaintTimer = false;
    repaintData->systemState = IDLE_STATE;
    repaintData->lastSubmittedSurface = NULL;
    wl_list_init(&repaintData->listFrameCallback);

    m_repaintData.insert(std::pair<uint, struct repaint_data*>(0, repaintData));
}

void* WaylandBaseWindowSystem::eventLoop()
{
    // INITALIZATION
    LOG_DEBUG("WaylandBaseWindowSystem", "Called, Enter thread");

    bool status = true;
    struct wl_event_loop *loop;

    do
    {
        m_wlDisplay = wl_display_create();

        if (NULL == m_wlDisplay)
        {
            LOG_ERROR("WaylandBaseWindowSystem",
                      "Failed to create wayland display");
            break;
        }
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Create wayland display, "
                  "display name=" << m_displayname << ", "
                  "wayland display=" << m_wlDisplay);

        loop = wl_display_get_event_loop(m_wlDisplay);
        wl_event_loop_add_signal(loop, SIGQUIT, signalEventOnTerm, this);
        wl_event_loop_add_signal(loop, SIGKILL, signalEventOnTerm, this);

        LOG_DEBUG("WaylandBaseWindowSystem", "wl_event_loop_add_signal");

        status = this->initCompositor();
        if (false == status)
        {
            LOG_ERROR("WaylandBaseWindowSystem",
                      "Failed to init compositor");
            break;
        }
        LOG_DEBUG("WaylandBaseWindowSystem", "SUCCESS:initCompositor");

        //. create Native Context
        status = createNativeContext();
        if (false == status)
        {
            LOG_ERROR("WaylandBaseWindowSystem", "Failed to create Native context");
            break;
        }
        LOG_DEBUG("WaylandBaseWindowSystem", "SUCCESS:createNativeContext");

        //. create egl context
        status = initGraphicSystem();
        if (false == status)
        {
            LOG_ERROR("WaylandBaseWindowSystem", "Failed to init graphicSystem");
            break;
        }
        LOG_DEBUG("WaylandBaseWindowSystem", "SUCCESS:init GraphicSystem");

        struct wl_event_loop *loop = wl_display_get_event_loop(m_wlDisplay);
        m_finishFrameTimer = wl_event_loop_add_timer(loop, finishFrameHandler, this);
        m_screenShotTimer = wl_event_loop_add_timer(loop, screenShotHandler, this);
        m_redrawCheckTimer = wl_event_loop_add_timer(loop, redrawCheckHandler, this);
        m_vnceventsflushTimer = wl_event_loop_add_timer(loop,vnceventsFlushHandler , this);
        m_fakeVysncSimulator = wl_event_loop_add_timer(loop, fakeVsyncHandler , this);
        /*Setup this timer to run cyclically every 16ms*/
        struct itimerspec timerSettings;
        /*start timer after a milli second*/
        timerSettings.it_value.tv_nsec = 1000;
        timerSettings.it_value.tv_sec = 0;
        /*Setup a vysnc interval corresponding to 60FPS*/
        timerSettings.it_interval.tv_nsec = VSYNC_INTERVAL_NS;
        timerSettings.it_interval.tv_sec = 0;
        timerfd_settime(m_fakeVysncSimulator->fd, 0, &timerSettings, NULL);

        initRepaintTimer();

        // Add idle callback that will post sem_run as soon as the main
        // loop is running to make sure we will only exit from ::start() after
        // we really started and the wayland socket is available
        wl_event_loop_add_idle(loop, idleEventStartupDone, this);

        // create input event
        status = createInputEvent();
        if (false == status)
        {
            LOG_WARNING("WaylandBaseWindowSystem", "WARNING: failed to create input event");
            status = true;
        }
        else
        {
            LOG_DEBUG("WaylandBaseWindowSystem", "SUCCESS:create InputEvent");
        }

        this->m_success = status;
        this->m_initialized = true;
        sem_post(&sem_run);

        // Done with init, wait for lock to actually run (ie start/stop method called)
        pthread_mutex_lock(&this->run_lock);

        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Starting Event loop ,"
                  "mutex lock=" << &this->run_lock << ", "
                  "display name=" << m_displayname);

        // run the main event loop while rendering
        gettimeofday(&tv0, NULL);
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Enter render loop, "
                  "mutex lock=" << &this->run_lock);

        if (wl_display_add_socket(m_wlDisplay, NULL))
        {
            LOG_ERROR("WaylandBaseWindowSystem", "failed to add socket");
            this->m_success = false;
            this->m_initialized = false;
            break;
        }

        // clear screen to avoid garbage on startup
        this->graphicSystem->clearBackground();
        this->graphicSystem->swapBuffers();

        wl_display_run(m_wlDisplay);
    } while (0);

    if (!this->m_success) sem_post(&sem_run);

    if(m_finishFrameTimer)
    {
        wl_event_source_remove(m_finishFrameTimer);
    }

    if(m_screenShotTimer)
    {
        wl_event_source_remove(m_screenShotTimer);
    }

    if(m_redrawCheckTimer)
    {
        wl_event_source_remove(m_redrawCheckTimer);
    }

    if(m_vnceventsflushTimer)
    {
        wl_event_source_remove(m_vnceventsflushTimer);
    }

    if(m_fakeVysncSimulator)
    {
        wl_event_source_remove(m_fakeVysncSimulator);
    }

    for (std::map<uint, struct repaint_data*>::iterator it = m_repaintData.begin();
         it != m_repaintData.end();
         it++)
    {
        struct repaint_data* repaintData;
        repaintData = it->second;

        if(repaintData->handleRepaintTimer)
        {
            wl_event_source_remove(repaintData->handleRepaintTimer);
        }

        delete it->second;
    }

    this->cleanup();
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Renderer thread finished, mutex lock=" << &this->run_lock);
    return NULL;
}

void WaylandBaseWindowSystem::signalRedrawEvent()
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, wayland display=" << m_wlDisplay << ", "
              "display name=" << m_displayname);
    // set flag that redraw is needed
    this->m_systemState = REDRAW_STATE;

    for (std::map<uint, struct repaint_data*>::iterator it = m_repaintData.begin();
          it != m_repaintData.end();
          it++)
    {
        struct repaint_data* repaintData;
        repaintData = it->second;
        repaintData->systemState = REDRAW_STATE;
    }

    wl_event_source_timer_update(m_finishFrameTimer, 1);
}

void WaylandBaseWindowSystem::cleanup()
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Cleanup, "
              "wayland display=" << m_wlDisplay << ", "
              "display name=" << m_displayname);

    shutdownCompositor();

    if (NULL != m_wlDisplay)
    {
        wl_display_destroy(m_wlDisplay);
        m_wlDisplay = NULL;
    }
}

static int add_systemd_sockets(struct wl_display* display)
{
    int fd;
    int cnt_systemd_sockets;
    int current_fd = 0;
    const char* runtime_dir;
    const char* name = "wayland-0";
    char path[256] = {0};

    runtime_dir = getenv("LM_SHARED_DIR");
    if (!runtime_dir) {
        LOG_DEBUG("WaylandBaseWindowSystem", "LM_SHARED_DIR not set in the environment");
        return 0;
    }

    snprintf(path, sizeof path, "%s/%s", runtime_dir, name);
    LOG_DEBUG("WaylandBaseWindowSystem", "Systemd socket should be in " << path);

    cnt_systemd_sockets = sd_listen_fds(0);

    if (cnt_systemd_sockets < 0) {
        LOG_ERROR("WaylandBaseWindowSystem",
                   "sd_listen_fds failed with:" <<cnt_systemd_sockets);
        return -1;
    }

    /* socket-based activation not used, return silently */
    if (cnt_systemd_sockets == 0)
        return 0;

    while (current_fd < cnt_systemd_sockets) {
        fd = SD_LISTEN_FDS_START + current_fd;

        if (sd_is_socket_unix(fd, AF_UNIX, 1, path, 0) <= 0) {
            LOG_DEBUG("WaylandBaseWindowSystem",
                      "invalid socket provided from systemd");
            current_fd++;
            continue;
        }

        if (wl_display_add_socket_fd(display, fd)) {
            LOG_ERROR("WaylandBaseWindowSystem",
                      "wl_display_add_socket_fd failed for systemd provided socket");
            return -1;
        }
        current_fd++;
    }

    LOG_DEBUG("WaylandBaseWindowSystem",
              "info: add " << current_fd << " "
              "socket(s) provided by systemd");

    return current_fd;
}

bool WaylandBaseWindowSystem::init(BaseGraphicSystem<void*, void*>* base)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, base=" << base << ", "
              "display name=" << m_displayname);
    graphicSystem = base;
    int status = pthread_create(&renderThread, NULL, eventLoopCallback, (void*)this);
    if (0 != status)
    {
        return false;
    }

    struct timespec timeout_time;
    clock_gettime(CLOCK_REALTIME, &timeout_time);
    timeout_time.tv_sec += 8; // Timeout after 8 seconds

    LOG_DEBUG("WaylandBaseWindowSystem",
          "Waiting start compositor complete");

    while (sem_timedwait(&sem_run, &timeout_time) == -1 && errno == EINTR)
    continue;

    if (m_initialized == false)
    {
        LOG_ERROR("WaylandBaseWindowSystem",
                  "Timed out waiting for compositor to start");
        return false;
    }

#ifdef WITH_WAYLAND_SYSTEMD_SYNC

        if (add_systemd_sockets(m_wlDisplay) < 0) {
            LOG_ERROR("WaylandBaseWindowSystem",
                      "add_systemd_sockets failed");
        }

        sd_notify(0, "READY=1");
#endif
    LOG_INFO("WaylandBaseWindowSystem",
              "Start complete [connect display], "
              "m_initialized=" << m_initialized << ", "
              "m_success=" << m_success << ", "
              "display name=" << m_displayname);

    
    return m_success;
}

bool WaylandBaseWindowSystem::start(int maxIterationDurationInMS)
{
    m_maxIterationDurationInMS = maxIterationDurationInMS;
    bool result = true;
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, maxIterationDurationInMS="
              << maxIterationDurationInMS << ", "
              "mutex lock=" << &this->run_lock << ", "
              "display name=" << m_displayname);
    // let thread actually run
    if (m_error == false)
    {
        pthread_mutex_unlock(&run_lock);
    }
    else
    {
        pthread_mutex_unlock(&run_lock);
        result = false;
    }

    // Wait on the event loop thread to actually start the wayland event loo
    // so that we are sure that the socket exists and wayland clients can
    // connect after ilm_init succeeds

    struct timespec ts;
    clock_gettime(CLOCK_REALTIME, &ts);
    ts.tv_nsec += 500 * 1000 * 1000;
    if (ts.tv_nsec >= 1000000000LL)
    {
        ts.tv_nsec -= 1000000000LL;
        ts.tv_sec++;
    }

    int s = 0;
    while ((s = sem_timedwait(&sem_run, &ts)) == -1 && errno == EINTR)
        continue;

    if (s == -1)
    {
        result = false;
        if (errno == ETIMEDOUT)
        {
            LOG_WARNING("WaylandBaseWindowSystem",
                        "Timeout waiting for wayland main loop to start");
        }
    }

    return result;
}

void WaylandBaseWindowSystem::stop()
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, mutex lock=" << &this->run_lock << ", "
              "display name=" << m_displayname);
    // needed if start was never called, we wake up thread, so it can immediatly finish
    // this->signalRedrawEvent();

    pthread_kill(renderThread, SIGQUIT);

    pthread_mutex_unlock(&run_lock);
    pthread_join(renderThread, NULL);

    // unblock a locked start
    sem_post(&sem_run);
}

void WaylandBaseWindowSystem::allocatePlatformSurface(Surface* surface)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, surface ID=" << surface->getID() << ", "
              "mutex lock=" << &this->run_lock << ", "
              "display name=" << m_displayname);
    WaylandPlatformSurface* nativeSurface = (WaylandPlatformSurface*)surface->platform;
    if (!nativeSurface)
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "Creating native surface for new window");
        // this surface does not have a native platform surface attached yet!
        nativeSurface = (WaylandPlatformSurface*)graphicSystem->getTextureBinder()->createPlatformSurface(surface);
        if (0 != nativeSurface)
        {
            unsigned int surfaceId = surface->getNativeContent();
            LOG_DEBUG("WaylandBaseWindowSystem",
                      "Got native surface, "
                      "surface->getNativeContent()=" << surfaceId << ", "
                      "surface ID=" << surface->getID() << ", "
                      "mutex lock=" << &this->run_lock << ", "
                      "wayland platform surface=" << nativeSurface);
            nativeSurface->connectionId = (unsigned short)((surfaceId >> 16) & 0xFFFF);
            nativeSurface->surfaceId = (unsigned short)(surfaceId & 0xFFFF);
            surface->platform = nativeSurface;
        }
        else
        {
            LOG_WARNING("WaylandBaseWindowSystem",
                        "Failed to allocate platformsurface"
                        "surface ID=" << surface->getID() << ", "
                        "mutex lock=" << &this->run_lock);
        }
    }
    else
    {
        LOG_DEBUG("WaylandBaseWindowSystem",
                  "wayland platform surface=" << nativeSurface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "mutex lock=" << &this->run_lock);
    }

    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
}

void WaylandBaseWindowSystem::deallocatePlatformSurface(Surface* surface)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, surface ID=" << surface->getID() << ", "
              "mutex lock=" << &this->run_lock << ", "
              "display name=" << m_displayname);
    WaylandPlatformSurface* nativeSurface = (WaylandPlatformSurface*)surface->platform;
    if (nativeSurface)
    {
        LOG_DEBUG("WaylandBaseWindowSystem", "destroying wayland platform surface");
        delete surface->platform;
        surface->platform = NULL;
    }
    LOG_DEBUG("WaylandBaseWindowSystem", "Exited");
}

void WaylandBaseWindowSystem::doScreenShot(std::string fileName, const uint screen_id)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, file name=" << fileName << ", "
              "screen ID=" << screen_id);
    m_takeScreenshot = ScreenshotOfDisplay;
    m_screenShotFile = fileName;
    m_screenShotScreenID = screen_id;
    wl_event_source_timer_update(m_screenShotTimer, 1);
}

void WaylandBaseWindowSystem::doScreenShotOfLayer(std::string fileName, const uint id)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, file name=" << fileName << ", "
              "Layer ID=" << id);
    m_takeScreenshot = ScreenshotOfLayer;
    m_screenShotFile = fileName;
    m_screenShotLayerID = id;
    wl_event_source_timer_update(m_screenShotTimer, 1);
}

void WaylandBaseWindowSystem::doScreenShotOfSurface(std::string fileName, const uint id, const uint layer_id)
{
    LOG_DEBUG("WaylandBaseWindowSystem",
              "Called, file name=" << fileName << ", "
              "Surface ID=" << id << ", "
              "Layer ID=" << layer_id);
    m_takeScreenshot = ScreenshotOfSurface;
    m_screenShotFile = fileName;
    m_screenShotSurfaceID = id;
    m_screenShotLayerID = layer_id;
    wl_event_source_timer_update(m_screenShotTimer, 1);
}

void WaylandBaseWindowSystem::manageWLInputEvent(const InputDevice type,
                                                InputEventState state,
                                                const WLEvent *wlEvent)
{
    LOG_INFO("WaylandBaseWindowSystem",
              "Called, device type=" << type << ", "
              "state=" << state << ", ");

    if (!m_inputEvent)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "InputEvent not available");
        return;
    }
    Surface *surface = NULL;
    native_surface *nativeSurface = NULL;
    uint32_t time = getTime();

     /* Locking scene mutex */
    m_pScene->lockScene();

    switch (type)
    {
    case INPUT_DEVICE_KEYBOARD:
        {
            LOG_DEBUG("WaylandBaseWindowSystem",
                        "INPUT_DEVICE_KEYBOARD: state=" << state << ", "
                        "keyCode=" << wlEvent->keyCode << ", "
                        "mutex lock=" << &this->run_lock);
            surface = m_pInputManager->reportKeyboardEvent(state, wlEvent->keyCode);
            if (!surface)
            {
                  LOG_WARNING("WaylandBaseWindowSystem",
                                "INPUT_DEVICE_KEYBOARD: key event with state=" << state << ", "
                                "keyCode=" << wlEvent->keyCode << " "
                                "is ignored as no surface has keyboard focus");
                m_inputEvent->inputDevice().setKeyboardFocus(NULL);
                break;
            }

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandBaseWindowSystem",
                            "KEYBOARD nativeSurface for surface, ID="<< surface->getID()
                            << " not found" << ", mutex lock=" << &this->run_lock);
                break;
            }
            LOG_INFO("WaylandBaseWindowSystem",
                      "INPUT_DEVICE_KEYBOARD: keycode "<< wlEvent->keyCode <<
                      "state: "<< state <<" send"
                      "to ilm surface ID="<< surface->getID() << ", "
                      "native surface="<< nativeSurface << ", "
                      "mutex lock=" << &this->run_lock);
            switch (state)
            {
            case INPUT_STATE_PRESSED:
                m_inputEvent->inputDevice().sendKeyPressEvent(
                    &nativeSurface->surface, time, wlEvent->keyCode);
                break;
            case INPUT_STATE_RELEASED:
                m_inputEvent->inputDevice().sendKeyReleaseEvent(
                    &nativeSurface->surface, time, wlEvent->keyCode);
                break;
            case INPUT_STATE_OTHER:
            default:
                // nothing to do
                break;
            }
        }
        break;
    case INPUT_DEVICE_POINTER:
        {
            LOG_DEBUG("WaylandBaseWindowSystem",
                      "INPUT_DEVICE_POINTER: state=" << state << ", "
                      "x=" << wlEvent->x << ", "
                      "y=" << wlEvent->y << ", "
                      "axis=" << wlEvent->axis << ", "
                      "value=" << wlEvent->axisValue << ", "
                      "mutex lock=" << &this->run_lock);
            Point globalPos = {state, wlEvent->x, wlEvent->y};
            Point localPos  = globalPos;
            surface = m_pInputManager->reportPointerEvent(localPos);
            if (!surface){
                LOG_WARNING("WaylandBaseWindowSystem",
                            "Surface Not Found!, "
                            "INPUT_DEVICE_POINTER: state=" << state << ", "
                            "axis=" << wlEvent->axis << ", "
                            "value=" << wlEvent->axisValue << ", "
                            "x=" << wlEvent->x << ", "
                            "y=" << wlEvent->y << ", "
                            "mutex lock=" << &this->run_lock);
                if (state == INPUT_STATE_PRESSED &&
                    m_inputEvent->inputDevice().pointerDevice() != NULL)
                {
                    m_inputEvent->inputDevice().pointerDevice()->button_count--;
                }

                break;
            }
            LOG_DEBUG("WaylandBaseWindowSystem",
                      "INPUT_DEVICE_POINTER: Local coordinates: "
                      "x=" << localPos.x << ", "
                      "y=" << localPos.y << ", "
                      "axis="<<  wlEvent->axis << ", "
                      "mutex lock=" << &this->run_lock);

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandBaseWindowSystem",
                            "INPUT_DEVICE_POINTER: "
                            "Native Surface not found for surface, "
                            "ID="<< surface->getID() << ", "
                            "mutex lock=" << &this->run_lock);

                if (state == INPUT_STATE_PRESSED &&
                    m_inputEvent->inputDevice().pointerDevice() != NULL)
                {
                    m_inputEvent->inputDevice().pointerDevice()->button_count--;
                }

                break;
            }
            LOG_INFO("WaylandBaseWindowSystem",
                      "INPUT_DEVICE_POINTER: "
                      "send event to ilm surface ID="<< surface->getID() << ", "
                      "state " << state << ", "
                      "local coord x=" << localPos.x << ", y=" << localPos.y << ", "
                      "axis " <<  wlEvent->axis <<", "
                      "axisValue " <<  wlEvent->axisValue <<", "
                      "time "<< time << ", "
                      "mutex lock=" << &this->run_lock);

            switch (state){
            case INPUT_STATE_PRESSED:
                m_inputEvent->inputDevice().sendMousePressEvent(
                    &nativeSurface->surface, globalPos, localPos, wlEvent->button, time);
                break;
            case INPUT_STATE_RELEASED:
                m_inputEvent->inputDevice().sendMouseReleaseEvent(
                    globalPos, localPos, wlEvent->button, time);
                break;
            case INPUT_STATE_MOTION:
                m_inputEvent->inputDevice().sendMouseMotionEvent(
                    &nativeSurface->surface, globalPos, localPos, time);
                break;
            case INPUT_STATE_AXIS:
                m_inputEvent->inputDevice().sendAxisEvent(&nativeSurface->surface,
                                                         wlEvent->axis,
                                                         wlEvent->axisValue,
                                                         time);
                break;
            case INPUT_STATE_OTHER:
            default:
                break;
            }
        }
        break;
    case INPUT_DEVICE_TOUCH:
        {
            LOG_DEBUG("WaylandBaseWindowSystem",
                        "INPUT_DEVICE_TOUCH: state=" << state << ", "
                        "x=" << wlEvent->x << ", "
                        "y=" << wlEvent->y << ", "
                        "mutex lock=" << &this->run_lock);
            if (wlEvent->touchId)
                state = INPUT_STATE_OTHER;

            Point pt = {state, wlEvent->x, wlEvent->y};
            PointVect ptVec(1, pt);
            surface = m_pInputManager->reportTouchEvent(ptVec);

            if (!surface) {
                if (m_inputEvent->inputDevice().touchDevice()->focus)
                {
                    m_inputEvent->inputDevice().cleanupTouchFocus();
                }
                break;
            }

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandBaseWindowSystem",
                            "INPUT_DEVICE_TOUCH: no native surface found "
                            "for surface, ID="<< surface->getID() << ", "
                            "mutex lock=" << &this->run_lock);
                break;
            }
            LOG_INFO("WaylandBaseWindowSystem",
                      "INPUT_DEVICE_TOUCH: "
                      "send to ilm surface ID=" << surface->getID() << ", "
                      "state "<< state <<", "
                      "touch id "<< wlEvent->touchId <<", "
                      "local coord x=" << ptVec[0].x << ", y=" << ptVec[0].y << ", "
                      "time "<< time );
            switch (state)
            {
            case INPUT_STATE_PRESSED:
            case INPUT_STATE_MOTION:
            case INPUT_STATE_RELEASED:
            case INPUT_STATE_OTHER:
                m_inputEvent->inputDevice().sendTouchPointEvent(
                    &nativeSurface->surface, time, wlEvent->touchId,
                    wlEvent->touchType, ptVec[0]);
                break;
            default:
                break;
            }
        }
        break;
    case INPUT_DEVICE_ALL:
    default:
        break;
    }

    /* Unlocking scene mutex */
    m_pScene->unlockScene();
}

void WaylandBaseWindowSystem::manageWLInputEvent(const InputDevice type, InputEventState state,
                                    const WLEvent *wlEvent, WaylandInputEvent *pWlIpEvent,
                                    char *pSeatName, int displayID)
{
    (void)pWlIpEvent;
    (void)pSeatName;
    (void)displayID;
    manageWLInputEvent(type, state, wlEvent);
}

ICommandExecutor& WaylandBaseWindowSystem::getCommandExecutor()
{
    return m_commandExecutor;
}

bool WaylandBaseWindowSystem::processVnctouchevent(int screenID,int x,int y,int down)
{
    WLEvent  wlEvent = { 0 };
    InputEventState eventState;
    unsigned int deviceType = INPUT_DEVICE_TOUCH;
    static int prev_down_state = 0;
    bool  ret_val = false;
    WaylandInputEvent* pWlinputEvent = NULL;
    const char *pseatname = NULL;
    WaylandEvdevInputEvent* inputEvent = (WaylandEvdevInputEvent *)m_inputEvent;

    wlEvent.x = x;
    wlEvent.y = y;

    if (inputEvent == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "m_inputEvent is NULL");
    }
    else
    {
        pWlinputEvent = inputEvent->getWaylandInputEvent(screenID);

        if(NULL == pWlinputEvent)
        {
            /* assign default input event*/
            pWlinputEvent = m_inputEvent;
        }

        pseatname = pWlinputEvent->getSeatName();
        if( pseatname == NULL)
        {
            /* assign default seat name */
            pseatname = m_inputEvent->getSeatName();
        }

        if (pWlinputEvent->inputDevice().hasTouch())
        {
            if(  (down == 1) && ( prev_down_state == 0) )
            {
                wlEvent.touchType = WL_TOUCH_DOWN;
                eventState = INPUT_STATE_PRESSED;
            }
            else if( ( prev_down_state == 1) && (down == 0))
            {

                wlEvent.touchType = WL_TOUCH_UP;
                eventState = INPUT_STATE_RELEASED;
                wlEvent.x = 0;
                wlEvent.y = 0;
            }
            else if(  (down == 1) && ( prev_down_state == 1) )
            {
                wlEvent.touchType = WL_TOUCH_MOTION;
                eventState = INPUT_STATE_MOTION;
            }
            else
            {
            	/* without any down event,touch motion event could not send*/
                return true;
            }

            LOG_DEBUG("WaylandBaseWindowSystem","touchType="<<wlEvent.touchType<<", "
                                "eventState="<<eventState<<", "
                                "down="<<down<<", "
                                "x="<<x <<" y= "<<y);

            prev_down_state = down;

            pWlinputEvent->windowSystem().manageWLInputEvent (deviceType,
                                    eventState,
                                    &wlEvent,
                                    pWlinputEvent,
                                    (char*)pseatname,
                                    screenID);

            /* used to dispatch the vnc sent events from wl_display_run */
            wl_event_source_timer_update(m_vnceventsflushTimer, 1);
            ret_val = true;
        }
    }
    return ret_val;
}

bool WaylandBaseWindowSystem::processVncmouseevent(int screenID,int x,int y,int buttonmask)
{
    bool ret = false;
    WLEvent wlEvent = { 0};
    InputEventState eventState = INPUT_STATE_MOTION;
    unsigned int deviceType = INPUT_DEVICE_POINTER;
    WaylandInputEvent* pWlinputEvent = NULL;
    const char *pseatname = NULL;
    WaylandEvdevInputEvent* inputEvent = (WaylandEvdevInputEvent *)m_inputEvent;

    LOG_DEBUG("WaylandBaseWindowSystem","called");

    static int prev_buttonmask = 0;

    wlEvent.x = x;
    wlEvent.y = y;

    if (inputEvent == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "m_inputEvent is NULL");
    }
    else
    {
        pWlinputEvent = inputEvent->getWaylandInputEvent(screenID);

        if(NULL == pWlinputEvent)
        {
            /* assign default input event*/
            pWlinputEvent = m_inputEvent;
        }

        pseatname = pWlinputEvent->getSeatName();
        if( pseatname == NULL)
        {
            /* assign default seat name */
            pseatname = m_inputEvent->getSeatName();
        }

        if (pWlinputEvent->inputDevice().hasPointer())
        {
            if(  (buttonmask == 1) && ( prev_buttonmask == 0) )
            {
                eventState = INPUT_STATE_PRESSED;
                wlEvent.touchType = WL_TOUCH_MOTION;
            }
            else if( ( prev_buttonmask == 1) && (buttonmask == 0))
            {
                eventState = INPUT_STATE_RELEASED;
                wlEvent.touchType = WL_TOUCH_MOTION;
            }
            prev_buttonmask = buttonmask;

            pWlinputEvent->windowSystem().manageWLInputEvent(deviceType,
                                eventState,
                                &wlEvent,
                                pWlinputEvent,
                                (char*)pseatname,
                                screenID);
            /* used to dispatch the vnc sent events from wl_display_run */
            wl_event_source_timer_update(m_vnceventsflushTimer, 1);
            ret = true;
        }
    }
    return ret;
}

bool WaylandBaseWindowSystem::processVnckeyboardevent(int screenID,unsigned int keycode, int down)
{
    WLEvent  wlEvent = { 0 };
    InputEventState state ;
    bool ret_val = false;
    WaylandInputEvent* pWlinputEvent = NULL;
    const char *pseatname = NULL;
    WaylandEvdevInputEvent* inputEvent = (WaylandEvdevInputEvent *)m_inputEvent;

    LOG_DEBUG("WaylandBaseWindowSystem","called screenID="<<screenID<<", keycode" <<keycode);

    if (inputEvent == NULL)
    {
        LOG_WARNING("WaylandBaseWindowSystem", "m_inputEvent is NULL");
    }
    else
    {
        pWlinputEvent = inputEvent->getWaylandInputEvent(screenID);

        if(NULL == pWlinputEvent)
        {
            /* assign default input event*/
            pWlinputEvent = m_inputEvent;
        }

        pseatname = pWlinputEvent->getSeatName();
        if( pseatname == NULL)
        {
            /* assign default seat name */
            pseatname = m_inputEvent->getSeatName();
        }

        if (m_inputEvent->inputDevice().hasKeyboard())
        {
            LOG_WARNING("WaylandBaseWindowSystem",
                        "INPUT_DEVICE_KEYBOARD: not configured");

            wlEvent.keyCode = keycode;

            if ( down)
            {
                wlEvent.keyState = WL_KEYBOARD_KEY_STATE_PRESSED;
                state   =   INPUT_STATE_PRESSED;
            }
            else
            {
                wlEvent.keyState = WL_KEYBOARD_KEY_STATE_RELEASED;
                state = INPUT_STATE_RELEASED;
            }

            pWlinputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_KEYBOARD,
                                state,
                                &wlEvent,
                                pWlinputEvent,
                                (char*)pseatname,
                                screenID);
            /* used to dispatch the vnc events from wl_display_run thread */
            wl_event_source_timer_update(m_vnceventsflushTimer, 1);
            ret_val =  true;
        }
    }
    return ret_val;
}

int WaylandBaseWindowSystem::vnceventsFlushHandler(void *data)
{
    (void)(data);
     /* 
        * This api is used to call wl_display_flush_clients which will be executed 
        *  from wl_display_run after this api.
        */
        return 1;
}

int WaylandBaseWindowSystem::getVncDisplayConnectionState(int displayID,
                                                                        void **pp_vncbuffaddr,void **pp_vncsurfptr)
{
    LOG_DEBUG("WaylandBaseWindowSystem","called");

    return ( m_commandExecutor.getVncDisplayConnectionState(displayID,\
                    pp_vncbuffaddr,pp_vncsurfptr) );
}

bool WaylandBaseWindowSystem::NotifyVnc_BeforeFramebufferUpdate(int displayID)
{
    LOG_DEBUG("WaylandBaseWindowSystem","called");

    return ( m_commandExecutor.NotifyVnc_BeforeFramebufferUpdate(displayID) );
}

bool WaylandBaseWindowSystem::NotifyVnc_FramebufferUpdated(int displayID)
{
	LOG_DEBUG("WaylandBaseWindowSystem","called");

	return ( m_commandExecutor.NotifyVnc_FramebufferUpdated(displayID) );
}
